/*  train command line
 *  Copyright (c) 2000 ARINO Kazuma & FURUKAWA Yo.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
/* $Id: train.c,v 1.1.1.1 2000/05/07 06:44:33 furukawa Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>

#include "libtrain.h"

static void
show_result(char *start, char *end)
{
    int startid, endid;

    startid = Train_SetStartPoint(start);
    endid = Train_SetEndPoint(end);
    Train_Search(startid, endid);
    
    printf("⤭=%s, Ԥ=%s, ֻ=%s, =%s\n",
	   Train_GetWalkText(),
	   Train_GetWaitText(),
	   Train_GetJoushaText(),
	   Train_GetTotalText());

    printf("%s", Train_GetResultText());
}

static void
show_prompt(void)
{
    printf("train> ");
}

#define iswhite(c) (c == ' ' || c == '\t')

static char *
get_argument(char *buf)
{
    char *p = buf;
    
    while (!iswhite(*p)) p++;
    while (iswhite(*p)) p++;
    p[strlen(p)-1] = '\0';
    return p;
}

#define CMD_SET_DATA_FILE "set-data-file"
#define CMD_SET_DEPARTURE_TIME "set-departure-time"
#define CMD_SET_START "set-start"
#define CMD_SET_END "set-end"
#define CMD_SEARCH "search"
#define CMD_DISPLAY_WALK_TIME "display-walk-time"
#define CMD_DISPLAY_WAIT_TIME "display-wait-time"
#define CMD_DISPLAY_JOUSHA_TIME "display-jousha-time"
#define CMD_DISPLAY_TOTAL_TIME "display-total-time"
#define CMD_DISPLAY_RESULT "display-result"
#define CMD_LIMIT_WALK "limit-walk"
#define CMD_LIMIT_WAIT "limit-wait"
#define CMD_LIMIT_CHANGE "limit-change"
#define CMD_USE_CHARGED_EXPRESS "use-charged-express"
#define CMD_USE_SHINKANSEN "use-shinkansen"
#define CMD_USE_MISC "use-misc"
#define CMD_INCLUDE_STARTING_WAIT "include-starting-wait"
#define CMD_DETAILED_OUTPUT "detailed-output"
#define CMD_DISPLAY_COMPLEMENT_CANDIDATES "display-complement-candidates"
#define CMD_QUIT "quit"

static int
is_command(char *str, char *command)
{
    return (strlen(str) >= strlen(command))
	&& (0 == strncmp(str, command, strlen(command)));
}

static int
to_bool(char *str)
{
    if (strcmp(str, "true") == 0){
	return 1;
    }else{
	return 0;
    }
}

static void
interpreter(void)
{
    char buf[BUFSIZ];
    int start = 0, end = 0;
    
    show_prompt();
    while (fgets(buf, BUFSIZ, stdin) != NULL){
	if (is_command(buf, CMD_SET_DATA_FILE)){
	    Train_SetDataFile(get_argument(buf));
	}else if (is_command(buf, CMD_SET_DEPARTURE_TIME)){
	    Train_SetDepartureTime(atoi(get_argument(buf)));
	}else if (is_command(buf, CMD_SET_START)){
	    start = Train_SetStartPoint(get_argument(buf));
	}else if (is_command(buf, CMD_SET_END)){
	    end = Train_SetEndPoint(get_argument(buf));
	}else if (is_command(buf, CMD_SEARCH)){
	    Train_Search(start, end);
	}else if (is_command(buf, CMD_DISPLAY_WALK_TIME)){
	    printf("%s\n", Train_GetWalkText());
	}else if (is_command(buf, CMD_DISPLAY_WAIT_TIME)){
	    printf("%s\n", Train_GetWaitText());
	}else if (is_command(buf, CMD_DISPLAY_JOUSHA_TIME)){
	    printf("%s\n", Train_GetJoushaText());
	}else if (is_command(buf, CMD_DISPLAY_TOTAL_TIME)){
	    printf("%s\n", Train_GetTotalText());
	}else if (is_command(buf, CMD_DISPLAY_RESULT)){
	    printf(Train_GetResultText());
	}else if (is_command(buf, CMD_LIMIT_WALK)){
	    Train_SetWalkLimit(atoi(get_argument(buf)));
	}else if (is_command(buf, CMD_LIMIT_WAIT)){
	    Train_SetWaitLimit(atoi(get_argument(buf)));
	}else if (is_command(buf, CMD_LIMIT_CHANGE)){
	    Train_SetChangeLimit(atoi(get_argument(buf)));
	}else if (is_command(buf, CMD_USE_CHARGED_EXPRESS)){
	    Train_SetUseChargedExpress(to_bool(get_argument(buf)));
	}else if (is_command(buf, CMD_USE_SHINKANSEN)){
	    Train_SetUseShinkansen(to_bool(get_argument(buf)));
	}else if (is_command(buf, CMD_USE_MISC)){
	    Train_SetUseMisc(to_bool(get_argument(buf)));
	}else if (is_command(buf, CMD_INCLUDE_STARTING_WAIT)){
	    Train_SetStartingWait(to_bool(get_argument(buf)));
	}else if (is_command(buf, CMD_DETAILED_OUTPUT)){
	    Train_SetDetailed(to_bool(get_argument(buf)));
	}else if (is_command(buf, CMD_DISPLAY_COMPLEMENT_CANDIDATES)){
	    int id, ss_len, cookie = 0;
	    char ss[BUFSIZ];
	    sprintf(ss, "%s", get_argument(buf));
	    ss_len = strlen(ss);
	    while ((id = Train_GetStationComplementCandidate(ss, ss_len, &cookie)) >= 0){
		printf("%s\n", Train_GetStationNameInRoman(id));
	    }
	}else if (is_command(buf, CMD_QUIT)){
	    exit(0);
	}else{
	    printf("ERROR!\n");
	}
	show_prompt();
    }

}

int main(int argc,char *argv[])
{
    char c;
    extern char *optarg;
    extern int optind;
    int interactive_mode = 0;

    Train_Initialize();

    while ((c = getopt(argc,argv,"d:i")) != EOF){
	switch(c){
	case 'd':
	    Train_SetDataFile(optarg);
	    break;
	case 'i':
	    interactive_mode = 1;
	    break;
	}
    }

    if (interactive_mode){
	show_result(argv[optind], argv[optind + 1]);
    }else{
	interpreter();
    }

    Train_Finalize();

    return 0;
}
