// ---------------------------------------------------------------------------
// - Locos.cpp                                                               -
// - afnix:csm module - local clous object set class implementation          -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2012 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Locos.hpp"
#include "Vector.hpp"
#include "Boolean.hpp"
#include "Integer.hpp"
#include "Runnable.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a nil set

  Locos::Locos (void) {
    reset ();
  }

  // return the locos class name

  String Locos::repr (void) const {
    return "Locos";
  }

  // reset this locos
  
  void Locos::reset (void) {
    wrlock ();
    try {
      d_cos.reset ();
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the number of rco in the locos

  long Locos::length (void) const {
    rdlock ();
    try {
      long result = d_cos.length ();
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }
	
  // return true if the locos is empty

  bool Locos::empty (void) const {
    rdlock ();
    try {
      bool result = d_cos.empty ();
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // check if an rco exists in this set

  bool Locos::exists (const Rco& rco) const {
    rdlock ();
    try {
      // get the set length
      long slen = d_cos.length ();
      // loop in the set
      for (long i = 0; i < slen; i++) {
	Object* obj = d_cos.get (i);
	Rco*    cro = dynamic_cast <Rco*> (obj);
	if (cro == nilp) {
	  throw Exception ("internal-error", "invalid object in locos",
			   Object::repr (obj));
	}
	// check and exists if needed
	bool result = (rco == *cro);
	if (result == true) {
	  unlock ();
	  return true;
	}
      }
      unlock ();
      return false;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // find the rco index in the locos

  long Locos::getidx (const Rco& rco) const {
    rdlock ();
    try {
      // get the set length
      long slen = d_cos.length ();
      // loop in the set
      for (long i = 0; i < slen; i++) {
	Object* obj = d_cos.get (i);
	Rco*    cro = dynamic_cast <Rco*> (obj);
	if (cro == nilp) {
	  throw Exception ("internal-error", "invalid object in locos",
			   Object::repr (obj));
	}
	// check and exists if needed
	bool result = (rco == *cro);
	if (result == true) {
	  unlock ();
	  return i;
	}
      }
      throw Exception ("locos-error", "cannot find rco index in locos");
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // add a rco in this locos

  void Locos::add (const Rco& rco) {
    wrlock ();
    try {
      if (exists (rco) == false) d_cos.add (new Rco (rco));
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get a rco object by index

  Rco Locos::get (const long index) const {
    rdlock ();
    try {
      Object* obj = d_cos.get (index);
      Rco*    rco = dynamic_cast <Rco*> (obj);
      if (rco == nilp) {
	throw Exception ("internal-error", "invalid object in locos",
			 Object::repr (obj));
      }
      Rco result = (*rco);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 6;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_ADD     = zone.intern ("add");
  static const long QUARK_GET     = zone.intern ("get");
  static const long QUARK_LENGTH  = zone.intern ("length");
  static const long QUARK_EMPTYP  = zone.intern ("empty-p");
  static const long QUARK_GETIDX  = zone.intern ("get-index");
  static const long QUARK_EXISTSP = zone.intern ("exists-p");

  // create a new object in a generic way

  Object* Locos::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // check for 0 argument
    if (argc == 0) return new Locos;
    throw Exception ("argument-error",
                     "too many argument with locos constructor");
  }

  // return true if the given quark is defined

  bool Locos::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Object::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }
  
  // apply this object with a set of arguments and a quark
  
  Object* Locos::apply (Runnable* robj, Nameset* nset, const long quark,
		      Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    
    // check for 0 argument
    if (quark == QUARK_EMPTYP) return new Boolean (empty  ());
    if (quark == QUARK_LENGTH) return new Integer (length ());
    // check for 1 argument
    if (argc == 1) {
      if (quark == QUARK_ADD) {
	Object* obj = argv->get (0);
	Rco*    rco = dynamic_cast <Rco*> (obj);
	if (rco == nilp) {
	  throw Exception ("type-error", "invalid object with locos add",
			   Object::repr (obj));
	}
	add (*rco);
	return nilp;
      }
      if (quark == QUARK_GET) {
	long index = argv->getlong (0);
	return new Rco (get (index));
      }
      if (quark == QUARK_GETIDX) {
	Object* obj = argv->get (0);
	Rco*    rco = dynamic_cast <Rco*> (obj);
	if (rco == nilp) {
	  throw Exception ("type-error", "invalid object with locos get-index",
			   Object::repr (obj));
	}
	return new Integer (getidx (*rco));
      }
      if (quark == QUARK_EXISTSP) {
	Object* obj = argv->get (0);
	Rco*    rco = dynamic_cast <Rco*> (obj);
	if (rco == nilp) {
	  throw Exception ("type-error", "invalid object with locos exists-p",
			   Object::repr (obj));
	}
	return new Boolean (exists (*rco));
      }
    }
    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
