/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (c) 2000-2003 QoSient, LLC
 * All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
 * raxml - writeout XML data format for argus data.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */

#include <argus_client.h>
#include <signal.h>

void RaXMLPrintFlowEncapsData(struct ArgusRecord *argus);
void RaXMLPrintMacData(struct ArgusRecord *argus);
void RaXMLPrintAddrNames(struct ArgusRecord *argus);
void RaXMLPrintFlowAttrsData(struct ArgusRecord *argus);
void RaXMLPrintMACAttrsData(struct ArgusRecord *argus);
void RaXMLPrintMetricsData (struct ArgusRecord *argus);
void RaXMLPrintTimeData (struct ArgusRecord *argus);
void RaXMLPrintUsrData (struct ArgusRecord *argus);
void RaXMLPrintAgrData (struct ArgusRecord *argus);

int RaInitialized = 0;
int RaStartedXML = 0;
int RaXMLRecords = 0;

void
ArgusClientInit ()
{
   if (!(RaInitialized)) {
      RaInitialized++;
      pflag = 0;
      ArgusPrintGMT = 0;

      if (ArgusPrintGMT)
         RaTimeFormat="%Y-%m-%dT%TZ";
      else
         RaTimeFormat="%Y-%m-%dT%T";

      (void) signal (SIGHUP,  (void (*)(int)) RaParseComplete);
      (void) signal (SIGTERM, (void (*)(int)) RaParseComplete);
      (void) signal (SIGQUIT, (void (*)(int)) RaParseComplete);
      (void) signal (SIGINT,  (void (*)(int)) RaParseComplete);
   }
}

void
RaParseComplete (int sig)
{
   if ((sig >= 0) && RaStartedXML) {
      printf("  <ArgusDataStream.End count = \"%d\" />\n", RaXMLRecords); 
      printf("</ArgusDataStream>\n");
      fflush(stdout);
   }

   _exit(0);
}

void
ArgusClientTimeout ()
{
}

void
parse_arg (int argc, char**argv)
{ 
}


void
usage ()
{
   extern char version[];
   fprintf (stderr, "Raxml Version %s\n", version);
   fprintf (stderr, "usage: %s \n", ArgusProgramName);
   fprintf (stderr, "usage: %s [ra-options] [- filter-expression]\n", ArgusProgramName);

   fprintf (stderr, "ra-options: -b                dump packet-matching code.\n");
   fprintf (stderr, "            -C                treat the remote source as a Cisco Netflow source.\n");
   fprintf (stderr, "            -D <level>        specify debug level\n");
   fprintf (stderr, "            -e <encode>       convert user data using <encode> method.\n");
   fprintf (stderr, "                              Supported types are <Ascii> and <Encode64>.\n");
   fprintf (stderr, "            -E <file>         write records that are rejected by the filter into <file>\n");
   fprintf (stderr, "            -F <conffile>     read configuration from <conffile>.\n");
   fprintf (stderr, "            -h                print help.\n");
   fprintf (stderr, "            -n                don't convert numbers to names.\n");
   fprintf (stderr, "            -r <file>         read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "            -S <host[:port]>  specify remote argus <host> and optional port number.\n");
   fprintf (stderr, "            -t <timerange>    specify <timerange> for reading records.\n");
   fprintf (stderr, "                     format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                              timeSpecification: [mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                                  mm/dd[/yy]\n");
   fprintf (stderr, "                                                  -%%d{yMhdms}\n");
   fprintf (stderr, "            -T <secs>         attach to remote server for T seconds.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "            -U <user/auth>    specify <user/auth> authentication information.\n");
#endif
   exit(1);
}


#include <stdlib.h>

void
RaProcessRecord (struct ArgusRecord *argus)
{
   struct timeval diffbuf, *diff = &diffbuf;
   char *StartDateBuf, *LastDateBuf;
   char *StartTimeBuf, *LastTimeBuf, *CauseStr;
   char IDStrBuf[32], *IDStr = IDStrBuf;

   if (!(RaStartedXML))
      RaProcessManRecord(&ArgusInput->ArgusManStart);

   if (argus->ahdr.type & ARGUS_MAR)
      RaProcessManRecord (argus);

   else {
      RaXMLRecords++;
      diff->tv_sec  = argus->argus_far.time.last.tv_sec -
                      argus->argus_far.time.start.tv_sec;

      diff->tv_usec = argus->argus_far.time.last.tv_usec -
                      argus->argus_far.time.start.tv_usec;

      if (diff->tv_usec < 0) {
         diff->tv_sec--;
         diff->tv_usec += 1000000;
      }

      RaTimeFormat="%Y-%m-%d";
      StartDateBuf = strdup(print_time(&argus->argus_far.time.start));
      RaTimeFormat="%T";
      StartTimeBuf = strdup(print_time(&argus->argus_far.time.start));

      RaTimeFormat="%Y-%m-%d";
      LastDateBuf = strdup(print_time(&argus->argus_far.time.last));
      RaTimeFormat="%T";
      LastTimeBuf = strdup(print_time(&argus->argus_far.time.last));

      switch (argus->ahdr.cause) {
         case ARGUS_START:    CauseStr = "Start"; break;
         case ARGUS_STATUS:   CauseStr = "Status"; break;
         case ARGUS_STOP:     CauseStr = "Stop"; break;
         case ARGUS_SHUTDOWN: CauseStr = "Shutdown"; break;
         case ARGUS_TIMEOUT:  CauseStr = "Timeout"; break;
         case ARGUS_ERROR:    CauseStr = "Error"; break;
         default:             CauseStr = "Unknown"; break;
      }

      if (argus->ahdr.status & ARGUS_ID_IS_IPADDR) {
         nflag++;
         IDStr = ipaddr_string(&argus->ahdr.argusid);
         nflag--;
      } else
         sprintf (IDStrBuf, "%u", argus->ahdr.argusid);

      printf("  <ArgusFlowRecord  ArgusSourceId = \"%s\" SequenceNumber = \"%d\" Cause = \"%s\"\n",
                                                 IDStr, argus->ahdr.seqNumber, CauseStr);
      printf("     StartDate = \"%s\" StartTime = \"%s\" StartTimeusecs = \"%d\"\n",
                                   StartDateBuf, StartTimeBuf, (int)argus->argus_far.time.start.tv_usec);
      printf("     LastDate  = \"%s\"  LastTime = \"%s\"  LastTimeusecs = \"%d\"\n",
                                    LastDateBuf,  LastTimeBuf, (int)argus->argus_far.time.last.tv_usec);
      printf("     Duration = \"%d.%06d\" TransRefNum =  \"%u\">\n", (int) diff->tv_sec, (int) diff->tv_usec,
                                                 argus->argus_far.ArgusTransRefNum);
      free(StartDateBuf);
      free(StartTimeBuf);
      free(LastDateBuf);
      free(LastTimeBuf);

      switch (argus->ahdr.status & 0xFFFF) {
         case ETHERTYPE_IP:
            switch (argus->argus_far.flow.ip_flow.ip_p) {
               case IPPROTO_TCP:
                  RaProcessTCPRecord (argus);
                  break;

               case IPPROTO_ICMP:
                  RaProcessICMPRecord (argus);
                  break;

               default:
               case IPPROTO_UDP:
                  RaProcessIPRecord (argus);
                  break;
            }
            break;

         case ETHERTYPE_ARP:
         case ETHERTYPE_REVARP:
            RaProcessARPRecord (argus);
            break;

         default:
            RaProcessNonIPRecord (argus);
            break;
      }
      printf("  </ArgusFlowRecord>\n");
      fflush(stdout);
   }
}


void
RaProcessManRecord (struct ArgusRecord *argus)
{
   extern struct ArgusInterfaceStruct interfacetypes [];
   extern char version[];
   struct ArgusInterfaceStruct *interface = &interfacetypes[0];
   char *StartDateBuf, *CurrentDateBuf, *StartTimeBuf, *CurrentTimeBuf;
   char IDStrBuf[32], *IDStr = IDStrBuf;
   
   if (argus->ahdr.cause == ARGUS_START) {
      if (RaStartedXML++ == 0) {
         Uflag = 0;
         XMLflag = 2;

      
         printf("<?xml version =\"1.0\"?>\n");
         printf("<!--Generated by raxml(%s) QoSient, LLC-->\n", version);

         RaTimeFormat="%Y-%m-%d";
         StartDateBuf = strdup(print_time(&argus->argus_far.time.start));
         RaTimeFormat="%T";
         StartTimeBuf = strdup(print_time(&argus->argus_far.time.start));

         RaTimeFormat="%Y-%m-%d";
         CurrentDateBuf = strdup(print_time(&argus->argus_far.time.start));
         RaTimeFormat="%T";
         CurrentTimeBuf = strdup(print_time(&argus->argus_far.time.start));

         printf("<ArgusDataStream \n");
         printf("  xmlns:xsi = \"http://www.w3.org/2001/XMLSchema-instance\"\n");
         printf("  xsi:noNamespaceSchemaLocation = \"http://qosient.com/argus/Xml/ArgusRecord.xsd\"\n");
         printf("  BeginDate   = \"%s\" BeginTime   = \"%s\" BeginTimeusec   = \"%d\"\n",
                          StartDateBuf, StartTimeBuf, (int)argus->argus_mar.startime.tv_usec);
         printf("  CurrentDate = \"%s\" CurrentTime = \"%s\" CurrentTimeusec = \"%d\"\n",
                          CurrentDateBuf, CurrentTimeBuf, (int)argus->argus_mar.now.tv_usec);
         printf("  MajorVersion = \"%d\" MinorVersion = \"%d\" ", major_version, minor_version);


         while (interface->value >= 0) {
            if (argus->argus_mar.interfaceType == interface->value)
               break;
            interface++;
         }

        nflag++;
         if (argus->ahdr.status & ARGUS_ID_IS_IPADDR) {
            IDStr = ipaddr_string(&argus->argus_mar.argusid);
         } else
            sprintf (IDStrBuf, "%u", argus->argus_mar.argusid);
   
         printf("InterfaceType = \"%s\" InterfaceStatus = \"%s\"\n", interface->label, "Up");
         printf("  SourceId = \"%s\" ", IDStr);
         printf(" NetAddr = \"%s\" ", ipaddr_string(&argus->argus_mar.localnet));
         printf(" NetMask = \"%s\">\n\n", ipaddr_string(&argus->argus_mar.netmask));
         nflag--;
         XMLflag = 1;
      }
   }

   fflush(stdout);
}


#define IPPROTOSTR 134
extern char *ip_proto_string [];
extern char *ArgusTCPFlags [];

#define ARGUSPROTOLEN	16
char ArgusProtoStrBuf[ARGUSPROTOLEN];

char *ArgusGetProtoString(unsigned short);

char *
ArgusGetProtoString(unsigned short proto)
{

   char *retn = NULL;

   bzero (ArgusProtoStrBuf, ARGUSPROTOLEN);

   if ((nflag > 1) || ((proto >= IPPROTOSTR) || !(strncmp("unas", ip_proto_string[proto], 4)))) {
      snprintf(ArgusProtoStrBuf, ARGUSPROTOLEN - 1, "%u", proto); 
      retn = ArgusProtoStrBuf;

   } else 
      retn = ip_proto_string[proto];
   
   return (retn);
}

void
RaProcessTCPRecord (struct ArgusRecord *argus)
{
   struct ArgusTCPObject *tcp = NULL;
   char SrcAddr[128], DstAddr[128], *protoStr = NULL;
   char TCPStatusStr[128], TCPOptionsStr[128], SrcTCPFlagsStr[16], DstTCPFlagsStr[16];
   struct ArgusFlow *flow = &argus->argus_far.flow;
   unsigned int status, options, i, index;

   nflag++;
   sprintf(SrcAddr, "%s", ipaddr_string(&flow->ip_flow.ip_src));
   sprintf(DstAddr, "%s", ipaddr_string(&flow->ip_flow.ip_dst));
   nflag--;
   
   protoStr = ArgusGetProtoString(flow->ip_flow.ip_p);

   if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS)
      tcp = (struct ArgusTCPObject *)ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX];

   bzero ((char *)TCPStatusStr, sizeof(TCPStatusStr));
   if ((tcp != NULL) && ((status = tcp->state) != 0)) {
      if (status) {
         if (status & ARGUS_SAW_SYN)         strcat (TCPStatusStr, "SYN|");
         if (status & ARGUS_SAW_SYN_SENT)    strcat (TCPStatusStr, "SYNACK|");
         if (status & ARGUS_CON_ESTABLISHED) strcat (TCPStatusStr, "EST|");
         if (status & ARGUS_FIN)             strcat (TCPStatusStr, "FIN|");
         if (status & ARGUS_FIN_ACK)         strcat (TCPStatusStr, "FINACK|");
         if (status & ARGUS_NORMAL_CLOSE)    strcat (TCPStatusStr, "CLO|");
         if (status & ARGUS_RESET)           strcat (TCPStatusStr, "RST|");

         if (status & ARGUS_SRC_WINDOW_SHUT) strcat (TCPStatusStr, "SWIN|");
         if (status & ARGUS_DST_WINDOW_SHUT) strcat (TCPStatusStr, "DWIN|");
         if (status & ARGUS_SRC_CONGESTED)   strcat (TCPStatusStr, "SECN|");
         if (status & ARGUS_DST_CONGESTED)   strcat (TCPStatusStr, "DECN|");
         if (status & ARGUS_SRC_PKTS_RETRANS)strcat (TCPStatusStr, "SRTN|");
         if (status & ARGUS_DST_PKTS_RETRANS)strcat (TCPStatusStr, "DRTN|");

         TCPStatusStr[strlen(TCPStatusStr) - 1] = '\0';

      } else
         strcat(TCPStatusStr, "None");
   } else
      strcat(TCPStatusStr, "None");

   bzero ((char *)TCPOptionsStr, sizeof(TCPOptionsStr));
   if ((tcp != NULL) && ((options = tcp->options) != 0)) {
      if (options) {
         if (options & ARGUS_TCP_MAXSEG)    strcat (TCPOptionsStr, "MAX|");
         if (options & ARGUS_TCP_WSCALE)    strcat (TCPOptionsStr, "WSC|");
         if (options & ARGUS_TCP_SACKOK)    strcat (TCPOptionsStr, "SACKOK|");
         if (options & ARGUS_TCP_SACK)      strcat (TCPOptionsStr, "SACK|");
         if (options & ARGUS_TCP_ECHO)      strcat (TCPOptionsStr, "ECHO|");
         if (options & ARGUS_TCP_ECHOREPLY) strcat (TCPOptionsStr, "ECHOR|");
         if (options & ARGUS_TCP_TIMESTAMP) strcat (TCPOptionsStr, "TIME|");
         if (options & ARGUS_TCP_CC)        strcat (TCPOptionsStr, "CC|");
         if (options & ARGUS_TCP_CCNEW)     strcat (TCPOptionsStr, "CCNEW|");
         if (options & ARGUS_TCP_CCECHO)    strcat (TCPOptionsStr, "CCECHO|");
         if (options & ARGUS_TCP_SRC_ECN)   strcat (TCPOptionsStr, "SECN|");
         if (options & ARGUS_TCP_DST_ECN)   strcat (TCPOptionsStr, "DECN|");

         TCPOptionsStr[strlen(TCPOptionsStr) - 1] = '\0';
      } else
         strcat(TCPOptionsStr, "None");
   } else
      strcat(TCPOptionsStr, "None");


   bzero(SrcTCPFlagsStr, sizeof(SrcTCPFlagsStr));
   bzero(DstTCPFlagsStr, sizeof(DstTCPFlagsStr));

   if (tcp != NULL) {
      for (i = 0, index = 1; i < 8; i++) {
         if (tcp->src.flags & index) {
            strcat (SrcTCPFlagsStr, ArgusTCPFlags[i]);
         }
         if (tcp->dst.flags & index) {
            strcat (DstTCPFlagsStr, ArgusTCPFlags[i]);
         }
         index <<= 1;
      }
   }

   RaXMLPrintFlowEncapsData(argus);
   RaXMLPrintMACAttrsData(argus);

   printf("    <Flow> <IP SrcIPAddr = \"%s\" DstIPAddr = \"%s\" Proto = \"%s\" ", SrcAddr, DstAddr, protoStr);

   if (argus->argus_far.flow.ip_flow.sport != 0xFFFF)
      printf("Sport = \"%d\" ", argus->argus_far.flow.ip_flow.sport);
   if (argus->argus_far.flow.ip_flow.dport != 0xFFFF)
      printf("Dport = \"%d\" ", argus->argus_far.flow.ip_flow.dport);

   if ((argus->argus_far.flow.ip_flow.ip_id != 0xFFFF) || (argus->argus_far.flow.ip_flow.ip_id != 0))
      printf("IpId = \"%x\" ", argus->argus_far.flow.ip_flow.ip_id);

   printf ("/> </Flow>\n");

   RaXMLPrintAddrNames(argus);
   RaXMLPrintFlowAttrsData(argus);

   if (tcp != NULL) {

      printf("    <ExtFlow> <TCPExtFlow TCPState = \"%s\" TCPOptions = \"%s\" SynAckuSecs = \"%u\" AckDatauSecs = \"%u\" >\n",
                              TCPStatusStr, TCPOptionsStr, tcp->synAckuSecs, tcp->ackDatauSecs);
      printf("              <TCPExtMetrics  SrcTCPSeqBase = \"%u\" SrcTCPAckBytes = \"%u\" ",
               tcp->src.seqbase, tcp->src.ackbytes);
      printf ("SrcTCPBytes = \"%u\" SrcTCPRetrans = \"%d\" SrcTCPWin = \"%d\" SrcTCPFlags = \"%s\"\n",
               tcp->src.bytes, tcp->src.rpkts, tcp->src.win, SrcTCPFlagsStr);
      printf("                              DstTCPSeqBase = \"%u\" DstTCPAckBytes = \"%u\" ",
               tcp->dst.seqbase, tcp->dst.ackbytes);
      printf ("DstTCPBytes = \"%u\" DstTCPRetrans = \"%d\" DstTCPWin = \"%d\" DstTCPFlags = \"%s\" />\n",
               tcp->dst.bytes, tcp->dst.rpkts, tcp->dst.win, DstTCPFlagsStr);
      printf("              </TCPExtFlow>\n");
      printf("    </ExtFlow>\n");
   }

   RaXMLPrintMetricsData(argus);
   RaXMLPrintTimeData(argus);
   RaXMLPrintUsrData(argus);
/*
   RaXMLPrintAgrData (argus);
*/
}

#if defined(__OpenBSD__)
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#endif

#include <netinet/ip_icmp.h>
extern char *icmptypestr[];

void
RaProcessICMPRecord (struct ArgusRecord *argus)
{
   char SrcAddr[128], DstAddr[128];
   char *protoStr, *typeStr = NULL, *codeStr = NULL;
   struct ArgusFlow *flow = &argus->argus_far.flow;
   struct ArgusICMPFlow *icmpFlow = &argus->argus_far.flow.icmp_flow;
   struct ArgusICMPObject *icmp = NULL;
   unsigned char ra_icmp_type = 0, ra_icmp_code = 0;
   unsigned short ra_icmp_data = 0;
   unsigned int  ra_src_addr = 0, ra_dst_addr = 0, ra_gw_addr = 0;

   
   nflag++;
   sprintf(SrcAddr, "%s", ipaddr_string(&flow->ip_flow.ip_src));
   sprintf(DstAddr, "%s", ipaddr_string(&flow->ip_flow.ip_dst));
   nflag--;

   protoStr = ArgusGetProtoString(flow->ip_flow.ip_p);

   if (ArgusThisFarStatus & ARGUS_ICMP_DSR_STATUS) {
      icmp = (struct ArgusICMPObject *)ArgusThisFarHdrs[ARGUS_ICMP_DSR_INDEX];
      ra_src_addr  = icmp->isrcaddr;
      ra_dst_addr  = icmp->idstaddr;
      ra_gw_addr   = icmp->igwaddr;
      ra_icmp_type = icmp->icmp_type;
      ra_icmp_code = icmpFlow->code;

   } else {
      ra_icmp_type = icmpFlow->type;
      ra_icmp_code = icmpFlow->code;
   }

   ra_icmp_data = icmpFlow->id;

   if (ra_icmp_type < (unsigned char) (ICMP_MAXTYPE + 1))
      typeStr = icmptypestr[ra_icmp_type];
   else
      typeStr = "UNK";

#ifndef ICMP_UNREACH_NET_UNKNOWN
#define ICMP_UNREACH_NET_UNKNOWN        6
#endif
               
#ifndef ICMP_UNREACH_HOST_UNKNOWN
#define ICMP_UNREACH_HOST_UNKNOWN       7
#endif

#ifndef ICMP_UNREACH_ISOLATED
#define ICMP_UNREACH_ISOLATED           8
#endif

#ifndef ICMP_UNREACH_NET_PROHIB
#define ICMP_UNREACH_NET_PROHIB         9
#endif

#ifndef ICMP_UNREACH_HOST_PROHIB
#define ICMP_UNREACH_HOST_PROHIB        10
#endif

#ifndef ICMP_UNREACH_TOSNET
#define ICMP_UNREACH_TOSNET             11
#endif

#ifndef ICMP_UNREACH_TOSHOST
#define ICMP_UNREACH_TOSHOST            12
#endif
 
#ifndef ICMP_UNREACH_FILTER_PROHIB
#define ICMP_UNREACH_FILTER_PROHIB      13
#endif

#ifndef ICMP_UNREACH_HOST_PRECEDENCE
#define ICMP_UNREACH_HOST_PRECEDENCE    14
#endif

#ifndef ICMP_UNREACH_PRECEDENCE_CUTOFF
#define ICMP_UNREACH_PRECEDENCE_CUTOFF  15
#endif

   switch (ra_icmp_type) {
      case ICMP_UNREACH:
         switch (ra_icmp_code) {
            case ICMP_UNREACH_NET:               codeStr = "Net"; break;
            case ICMP_UNREACH_HOST:              codeStr = "Host"; break;
            case ICMP_UNREACH_PROTOCOL:          codeStr = "Proto"; break;
            case ICMP_UNREACH_PORT:              codeStr = "Port"; break;
            case ICMP_UNREACH_NEEDFRAG:          codeStr = "NeedFrag"; break;
            case ICMP_SR_FAILED:                 codeStr = "SrcRoute"; break;
            case ICMP_UNREACH_NET_UNKNOWN:       codeStr = "Net Unkn"; break;
            case ICMP_UNREACH_HOST_UNKNOWN:      codeStr = "Host Unkn"; break;
            case ICMP_UNREACH_ISOLATED:          codeStr = "Isolated"; break;
            case ICMP_UNREACH_NET_PROHIB:        codeStr = "Net Prohib"; break;
            case ICMP_UNREACH_HOST_PROHIB:       codeStr = "Host Prohib"; break;
            case ICMP_UNREACH_TOSNET:            codeStr = "Tos Net"; break;
            case ICMP_UNREACH_TOSHOST:           codeStr = "Tos Host"; break;
            case ICMP_UNREACH_FILTER_PROHIB:     codeStr = "Filter"; break;
            case ICMP_UNREACH_HOST_PRECEDENCE:   codeStr = "Host Prec"; break;
            case ICMP_UNREACH_PRECEDENCE_CUTOFF: codeStr = "Prec Xoff"; break;
         }
         break;

      case ICMP_REDIRECT:
         switch (ra_icmp_code) {
            case ICMP_REDIRECT_NET:              codeStr = "Net"; break;
            case ICMP_REDIRECT_HOST:             codeStr = "Host"; break;
            case ICMP_REDIRECT_TOSNET:           codeStr = "Tos Net"; break;
            case ICMP_REDIRECT_TOSHOST:          codeStr = "Tos Host"; break;
         }
         break;

      case ICMP_ROUTERADVERT:
         codeStr = ra_icmp_code ? "not_common" : "normal";
         break;

      case ICMP_TIMXCEED:
         codeStr = ra_icmp_code ? "reassembly" : "in_transit";
         break;

      case ICMP_PARAMETERPROB:
         codeStr = ra_icmp_code ? ((ra_icmp_code == 1) ? "option" : "length") : "pointer";
         break;
   }

   RaXMLPrintFlowEncapsData(argus);
   RaXMLPrintMACAttrsData(argus);

   printf("    <Flow> <ICMP SrcIPAddr = \"%s\" DstIPAddr = \"%s\" Proto = \"%s\" IcmpType = \"%s\" ",
         SrcAddr, DstAddr, protoStr, typeStr);

   if (codeStr)
      printf("IcmpCode = \"%s\" ", codeStr);

   printf("IcmpData = \"%d\"/> ", ra_icmp_data);

   if ((argus->argus_far.flow.icmp_flow.ip_id != 0xFFFF) || (argus->argus_far.flow.icmp_flow.ip_id != 0))
      printf("IpId = \"%x\" ", argus->argus_far.flow.icmp_flow.ip_id);

   printf("</Flow>\n");

   RaXMLPrintAddrNames(argus);
   RaXMLPrintFlowAttrsData(argus);
   RaXMLPrintMetricsData(argus);
   RaXMLPrintTimeData(argus);
   RaXMLPrintUsrData(argus);
   RaXMLPrintAgrData (argus);
}


void
RaProcessIPRecord (struct ArgusRecord *argus)
{
   char SrcAddr[128], DstAddr[128], *protoStr = NULL;
   struct ArgusFlow *flow = &argus->argus_far.flow;

   nflag++;
   sprintf(SrcAddr, "%s", ipaddr_string(&flow->ip_flow.ip_src));
   sprintf(DstAddr, "%s", ipaddr_string(&flow->ip_flow.ip_dst));
   nflag--;

   protoStr = ArgusGetProtoString(flow->ip_flow.ip_p);

   RaXMLPrintFlowEncapsData(argus);
   RaXMLPrintMACAttrsData(argus);

   if (ArgusThisFarStatus & ARGUS_ESP_DSR_STATUS) {
      printf("    <Flow> <Esp SrcIPAddr = \"%s\" DstIPAddr = \"%s\" Proto = \"%s\" ", SrcAddr, DstAddr, protoStr);
      if (argus->argus_far.flow.esp_flow.spi)
         printf("Spi = \"%d\" ", argus->argus_far.flow.esp_flow.spi);

   } else
      printf("    <Flow> <IP SrcIPAddr = \"%s\" DstIPAddr = \"%s\" Proto = \"%s\" ", SrcAddr, DstAddr, protoStr);

   if (flow->ip_flow.ip_p == IPPROTO_UDP) {
      if (argus->argus_far.flow.ip_flow.sport != 0xFFFF)
         printf("Sport = \"%d\" ", argus->argus_far.flow.ip_flow.sport);
      if (argus->argus_far.flow.ip_flow.dport != 0xFFFF)
         printf("Dport = \"%d\" ", argus->argus_far.flow.ip_flow.dport);
   }

   if ((argus->argus_far.flow.ip_flow.ip_id != 0xFFFF) || (argus->argus_far.flow.ip_flow.ip_id != 0))
      printf("IpId = \"%x\" ", argus->argus_far.flow.ip_flow.ip_id);

   printf ("/> </Flow>\n");

   RaXMLPrintAddrNames(argus);
   RaXMLPrintFlowAttrsData(argus);

   if (ArgusThisFarStatus & ARGUS_ESP_DSR_STATUS) {
      struct ArgusESPStruct *esp = (struct ArgusESPStruct *) ArgusThisFarHdrs[ARGUS_ESP_DSR_INDEX];
      char ArgusLastSeqBuf[32], ArgusLostSeqBuf[32];

      bzero(ArgusLastSeqBuf, 32);
      bzero(ArgusLostSeqBuf, 32);

      sprintf (ArgusLastSeqBuf, "\"%u\"", esp->src.lastseq);
      sprintf (ArgusLostSeqBuf, "\"%u\"", esp->src.lostseq);
      printf("       <ExtFlow> <EspExtFlow SrcEspLastSeq = %-9s SrcEspDroppedPkts = %s\n",    ArgusLastSeqBuf, ArgusLostSeqBuf);

      bzero(ArgusLastSeqBuf, 32);
      bzero(ArgusLostSeqBuf, 32);

      sprintf (ArgusLastSeqBuf, "\"%u\"", esp->dst.lastseq);
      sprintf (ArgusLostSeqBuf, "\"%u\"", esp->dst.lostseq);
      printf("                               DstEspLastSeq = %-9s DstEspDroppedPkts = %s />\n", ArgusLastSeqBuf, ArgusLostSeqBuf);
      printf("       </ExtFlow>\n");
   }

   if (ArgusThisFarStatus & ARGUS_FRG_DSR_STATUS) {
      struct ArgusFragObject *frag = (struct ArgusFragObject *) ArgusThisFarHdrs[ARGUS_FRG_DSR_INDEX];
 
      printf("       <ExtFlow> <FragExtFlow FragID = \"%d\" ExpectedLength = \"%d\" MaxFragLength = \"%d\" />",
                               frag->frag_id, frag->totlen, frag->maxfraglen);
      printf("</ExtFlow>\n");
   }
 
   RaXMLPrintMetricsData(argus);
   RaXMLPrintTimeData(argus);
   RaXMLPrintUsrData(argus);
   RaXMLPrintAgrData (argus);
}


#include <argus_filter.h>

void
RaProcessARPRecord (struct ArgusRecord *argus)
{
   char SrcAddr[128], TarAddr[128], MacAddr[128];
   struct ArgusArpFlow *arpFlow = &argus->argus_far.flow.arp_flow;

   nflag++;
   sprintf(SrcAddr, "%s", ipaddr_string(&arpFlow->arp_spa));
   sprintf(TarAddr, "%s", ipaddr_string(&arpFlow->arp_tpa));
   sprintf(MacAddr, "%s", etheraddr_string(arpFlow->etheraddr));
   nflag--;

   RaXMLPrintFlowEncapsData(argus);

   printf("    <Flow> <Arp SrcArpAddr = \"%s\" TargetArpAaddr = \"%s\" ArpMacAddr = \"%s\" /> </Flow>\n",
                                              SrcAddr, TarAddr, MacAddr);
 
   RaXMLPrintAddrNames(argus);
   RaXMLPrintMetricsData(argus);
   RaXMLPrintTimeData(argus);
   RaXMLPrintUsrData(argus);
   RaXMLPrintAgrData (argus);
}


void
RaProcessNonIPRecord (struct ArgusRecord *argus)
{
   char SrcAddr[128], DstAddr[128], *protoStr;
   struct ArgusMACFlow *macFlow = &argus->argus_far.flow.mac_flow;
   unsigned short proto;

   sprintf(SrcAddr, "%s", etheraddr_string((unsigned char *)&macFlow->ehdr.ether_shost));
   sprintf(DstAddr, "%s", etheraddr_string((unsigned char *)&macFlow->ehdr.ether_dhost));

   proto = argus->ahdr.status & 0xFFFF;
   protoStr = etherproto_string(proto);

   RaXMLPrintFlowEncapsData(argus);

   printf("    <Flow> <Mac SrcMacAddr = \"%s\" DstMacAddr = \"%s\" NetProto = \"%s\" /> </Flow>\n",
                                              SrcAddr, DstAddr, protoStr);
 
   RaXMLPrintMetricsData (argus);
   RaXMLPrintTimeData (argus);
   RaXMLPrintUsrData (argus);
   RaXMLPrintAgrData (argus);
}


int RaSendArgusRecord(struct ArgusRecordStore *argus) {return 0;}

void
RaXMLPrintFlowEncapsData(struct ArgusRecord *argus)
{
   if (ArgusThisFarStatus & ARGUS_VLAN_DSR_STATUS) {
      struct ArgusVlanStruct *vlan = (struct ArgusVlanStruct *) ArgusThisFarHdrs[ARGUS_VLAN_DSR_INDEX];

      printf("    <FlowEncaps> <VLAN Type = \"802.1Q\" ");
      if (vlan->status & ARGUS_SRC_VLAN) 
         printf("SrcVid = \"%x\" ", vlan->sid);
      if (vlan->status & ARGUS_DST_VLAN) 
         printf("DstVid = \"%x\" ", vlan->did);
      printf("/> </FlowEncaps>\n");
   }

   if (ArgusThisFarStatus & ARGUS_MPLS_DSR_STATUS) {
      struct ArgusMplsStruct *mpls = (struct ArgusMplsStruct *) ArgusThisFarHdrs[ARGUS_MPLS_DSR_INDEX];

      printf("    <FlowEncaps> <MPLS ");
      if (mpls->status & ARGUS_SRC_MPLS) 
         printf("SrcLabel = \"%x\" ", mpls->slabel);
      if (mpls->status & ARGUS_DST_MPLS) 
         printf("DstLabel = \"%x\" ", mpls->dlabel);
      printf("/> </FlowEncaps>\n");
   }
}


void
RaXMLPrintAddrNames(struct ArgusRecord *argus)
{
   if (!nflag) {
      char SrcAddr[128], DstAddr[128];

      switch (argus->ahdr.status & 0xFFFF) {
         case ETHERTYPE_IP: {
            struct ArgusFlow *flow = &argus->argus_far.flow;
            sprintf(SrcAddr, "%s", ipaddr_string(&flow->ip_flow.ip_src));
            sprintf(DstAddr, "%s", ipaddr_string(&flow->ip_flow.ip_dst));
            printf("    <IPAddrNames SrcName = \"%s\" DstName = \"%s\" />\n", SrcAddr, DstAddr);
            break;
         }

         case ETHERTYPE_ARP:
         case ETHERTYPE_REVARP: {
            struct ArgusArpFlow *arpFlow = &argus->argus_far.flow.arp_flow;

            sprintf(SrcAddr, "%s", ipaddr_string(&arpFlow->arp_spa));
            sprintf(DstAddr, "%s", ipaddr_string(&arpFlow->arp_tpa));
            printf("       <IPAddrNames SrcName = \"%s\" TargetName = \"%s\" />\n", SrcAddr, DstAddr);
            break;
         }

         default:
            return;
      }
   }
}



void
RaXMLPrintFlowAttrsData(struct ArgusRecord *argus)
{
   char *StatusStr = NULL, statusStrBuf[256];
   struct ArgusFlow *flow = &argus->argus_far.flow;
   int len;

   bzero (statusStrBuf, 256);
   StatusStr = statusStrBuf;

   if (argus->argus_far.status & ARGUS_FRAGMENTS)
      strcat(statusStrBuf, "Frag|");

   if (flow->ip_flow.tp_p == ARGUS_RTP_FLOWTAG)
      strcat(statusStrBuf, "RTP|");

   if (argus->ahdr.status & ARGUS_PPPoE)
      strcat(statusStrBuf, "PPPoE|");

   if (argus->argus_far.status & ARGUS_ICMP_MAPPED)
      strcat(statusStrBuf, "ICMP_MAPPED|");

   if (argus->ahdr.status & ARGUS_MULTIADDR)
      strcat(statusStrBuf, "MULTIADDR|");

   if ((len = strlen(StatusStr)) > 0)
      statusStrBuf[strlen(StatusStr) - 1] = '\0';
   else
      StatusStr = NULL;

   if (StatusStr)      
      printf("    <FlowAttrs Status = \"%s\" SrcTTL = \"%d\" DstTTL = \"%d\" SrcTOS = \"%x\" DstTOS = \"%x\" />\n", 
        StatusStr, argus->argus_far.attr_ip.sttl, argus->argus_far.attr_ip.dttl, argus->argus_far.attr_ip.stos, argus->argus_far.attr_ip.dtos);
   else
      printf("    <FlowAttrs SrcTTL = \"%d\" DstTTL = \"%d\" SrcTOS = \"%x\" DstTOS = \"%x\" />\n", 
        argus->argus_far.attr_ip.sttl, argus->argus_far.attr_ip.dttl, argus->argus_far.attr_ip.stos, argus->argus_far.attr_ip.dtos);
}


void
RaXMLPrintMACAttrsData(struct ArgusRecord *argus)
{
   char *esrcString = NULL, *edstString = NULL;

   if (ArgusThisFarStatus & ARGUS_MAC_DSR_STATUS) {
      struct ArgusMacStruct *mac = (struct ArgusMacStruct *) ArgusThisFarHdrs[ARGUS_MAC_DSR_INDEX];
 
      esrcString = etheraddr_string ((u_char *)&mac->phys_union.ether.ethersrc);
      edstString = etheraddr_string ((u_char *)&mac->phys_union.ether.etherdst);
 
      printf("    <MACAddrs SrcAddr = \"%s\" DstAddr = \"%s\" />\n", 
                       esrcString, edstString);
   }
}

void
RaXMLPrintMetricsData (struct ArgusRecord *argus)
{
   printf("    <Metrics SrcCount = \"%d\" DstCount = \"%d\" SrcBytes = \"%d\" DstBytes = \"%d\" ",
      argus->argus_far.src.count, argus->argus_far.dst.count, argus->argus_far.src.bytes, argus->argus_far.dst.bytes);
   printf(" SrcAppBytes = \"%d\" DstAppBytes = \"%d\" />\n",
             argus->argus_far.src.appbytes, argus->argus_far.dst.appbytes);
}

void
RaXMLPrintTimeData (struct ArgusRecord *argus)
{
   if (ArgusThisFarStatus & ARGUS_TIME_DSR_STATUS) {
      struct ArgusTimeStruct *time = (struct ArgusTimeStruct *) ArgusThisFarHdrs[ARGUS_TIME_DSR_INDEX];

      if ((time->src.act.n > 1)  || (time->dst.act.n > 1) ||
          (time->src.idle.n > 1) || (time->dst.idle.n > 1)) {
         printf ("       <ArgusTimeStats>\n");
         if (time->src.act.n) {
            printf ("          <SrcActive n = \"%d\" MeanUsec = \"%d\" StdDev = \"%d\" MaxUsec = \"%d\" MinUsec = \"%d\" />\n",
                time->src.act.n, time->src.act.meanval, time->src.act.stdev, time->src.act.maxval, time->src.act.minval);
         }
         if (time->src.idle.n) {
            printf ("          <SrcIdle   n = \"%d\" MeanUsec = \"%d\" StdDev = \"%d\" MaxUsec = \"%d\" MinUsec = \"%d\" />\n",
                   time->src.idle.n, time->src.idle.meanval, time->src.idle.stdev, time->src.idle.maxval, time->src.idle.minval);
         }
         if (time->dst.act.n) {
            printf ("          <DstActive n = \"%d\" MeanUsec = \"%d\" StdDev = \"%d\" MaxUsec = \"%d\" MinUsec = \"%d\" />\n",
                time->dst.act.n, time->dst.act.meanval, time->dst.act.stdev, time->dst.act.maxval, time->dst.act.minval);
         }
         if (time->dst.idle.n) {
            printf ("          <DstIdle   n = \"%d\" MeanUsec = \"%d\" StdDev = \"%d\" MaxUsec = \"%d\" MinUsec = \"%d\" />\n",
                   time->dst.idle.n, time->dst.idle.meanval, time->dst.idle.stdev, time->dst.idle.maxval, time->dst.idle.minval);
         }
         printf ("       </ArgusTimeStats>\n");
      }
   }
}

void
RaXMLPrintUsrData (struct ArgusRecord *argus)
{
   int len = 0;
   struct ArgusUserStruct *user = NULL;
   char strbuf[MAXSTRLEN], *str = strbuf;

   if (dflag && (ArgusThisFarStatus & (ARGUS_SRCUSRDATA_DSR_STATUS | ARGUS_DSTUSRDATA_DSR_STATUS))) {
      printf ("       <ArgusUserData>\n");
      if (ArgusThisFarStatus & ARGUS_SRCUSRDATA_DSR_STATUS) {
         user = (struct ArgusUserStruct *) ArgusThisFarHdrs[ARGUS_SRCUSRDATA_DSR_INDEX];

         len = (user->length - 1) * 4;
         len = (len < argus->argus_far.src.appbytes) ? len : argus->argus_far.src.appbytes;
         len = len > ArgusSrcUserDataLen ? ArgusSrcUserDataLen : len;

         if (ArgusSrcUserDataLen > 0) {
            printf ("          <SrcData Length = \"%d\" ", len);
            if ((len = ArgusEncode (&user->data, NULL, len, str, sizeof(strbuf))) != 0)
               printf ("%s />\n", str);
         }
      }

      if (ArgusThisFarStatus & ARGUS_DSTUSRDATA_DSR_STATUS) {
         struct ArgusUserStruct *user = (struct ArgusUserStruct *) ArgusThisFarHdrs[ARGUS_DSTUSRDATA_DSR_INDEX];
         char strbuf[MAXSTRLEN], *str = strbuf;

         len = (user->length - 1) * 4;
         len = (len < argus->argus_far.dst.appbytes) ? len : argus->argus_far.dst.appbytes;
         len = len > ArgusDstUserDataLen ? ArgusDstUserDataLen : len;

         printf ("          <DstData Length = \"%d\" ", len);
         if (ArgusEncode (&user->data, NULL, len, str, sizeof(strbuf)))
            printf ("%s />\n", str);
      }
      printf ("       </ArgusUserData>\n");
   }
}


void
RaXMLPrintAgrData (struct ArgusRecord *argus)
{
   struct ArgusAGRStruct *agr;

   if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) != NULL) {
      printf ("       <ArgusAgrData>\n");
      printf ("          <Count = \"%d\"\n", agr->count);
      if ((agr->act.n > 1) || (agr->idle.n > 1)) {
         if (agr->act.n) {
            if (agr->status & ARGUS_AGR_USECACTTIME) {
               printf ("                  <Packet n = \"%d\" MeanUsec = \"%d\" StdDev = \"%d\" MaxUsec = \"%d\" MinUsec = \"%d\" />\n",
                                                                  agr->act.n, agr->act.meanval, agr->act.stdev, agr->act.maxval, agr->act.minval);
            } else {
               printf ("                  <Packet n = \"%d\" MeanMsec = \"%d\" StdDev = \"%d\" MaxMsec = \"%d\" MinMsec = \"%d\" />\n",
                                                                  agr->act.n, agr->act.meanval, agr->act.stdev, agr->act.maxval, agr->act.minval);
            }
         }
         if (agr->idle.n) {
            if (agr->status & ARGUS_AGR_USECACTTIME) {
               printf ("             <Transaction n = \"%d\" MeanUsec = \"%d\" StdDev = \"%d\" MaxUsec = \"%d\" MinUsec = \"%d\" />\n",
                                                                  agr->idle.n, agr->idle.meanval, agr->idle.stdev, agr->idle.maxval, agr->idle.minval);
            } else {
               printf ("             <Transaction n = \"%d\" MeanMsec = \"%d\" StdDev = \"%d\" MaxMsec = \"%d\" MinMsec = \"%d\" />\n",
                                                                  agr->idle.n, agr->idle.meanval, agr->idle.stdev, agr->idle.maxval, agr->idle.minval);
            }
         }
      }
      printf ("       </ArgusAgrData>\n");
   }
}
