// Copyright 2023 The gVisor Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package metricserver

import (
	"crypto/sha256"
	"encoding/binary"
	"io"
	"strconv"

	"gvisor.dev/gvisor/pkg/prometheus"
	"gvisor.dev/gvisor/runsc/container"
)

// Metrics generated by the metrics server itself.
var (
	SandboxPresenceMetric = prometheus.Metric{
		Name: "sandbox_presence",
		Type: prometheus.TypeGauge,
		Help: "Boolean metric set to 1 for each known sandbox.",
	}
	SandboxRunningMetric = prometheus.Metric{
		Name: "sandbox_running",
		Type: prometheus.TypeGauge,
		Help: "Boolean metric set to 1 for each running sandbox.",
	}
	SandboxMetadataMetric = prometheus.Metric{
		Name: "sandbox_metadata",
		Type: prometheus.TypeGauge,
		Help: "Key-value pairs about per-sandbox metadata.",
	}
	SandboxCapabilitiesMetric = prometheus.Metric{
		Name: "sandbox_capabilities",
		Type: prometheus.TypeGauge,
		Help: "Linux capabilities added within containers of the sandbox.",
	}
	SandboxCapabilitiesMetricLabel = "capability"
	SpecMetadataMetric             = prometheus.Metric{
		Name: "spec_metadata",
		Type: prometheus.TypeGauge,
		Help: "Key-value pairs about OCI spec metadata.",
	}
	SandboxCreationMetric = prometheus.Metric{
		Name: "sandbox_creation_time_seconds",
		Type: prometheus.TypeGauge,
		Help: "When the sandbox was created, as a unix timestamp in seconds.",
	}
	NumRunningSandboxesMetric = prometheus.Metric{
		Name: "num_sandboxes_running",
		Type: prometheus.TypeGauge,
		Help: "Number of sandboxes running at present.",
	}
	NumCannotExportSandboxesMetric = prometheus.Metric{
		Name: "num_sandboxes_broken_metrics",
		Type: prometheus.TypeGauge,
		Help: "Number of sandboxes from which we cannot export metrics.",
	}
	NumTotalSandboxesMetric = prometheus.Metric{
		Name: "num_sandboxes_total",
		Type: prometheus.TypeCounter,
		Help: "Counter of sandboxes that have ever been started.",
	}
)

// Metrics is a list of metrics that the metric server generates.
var Metrics = []*prometheus.Metric{
	&SandboxPresenceMetric,
	&SandboxRunningMetric,
	&SandboxMetadataMetric,
	&SandboxCapabilitiesMetric,
	&SpecMetadataMetric,
	&SandboxCreationMetric,
	&NumRunningSandboxesMetric,
	&NumCannotExportSandboxesMetric,
	&NumTotalSandboxesMetric,
	&prometheus.ProcessStartTimeSeconds,
}

// SandboxPrometheusLabels returns a set of Prometheus labels that identifies the sandbox running
// the given root container.
func SandboxPrometheusLabels(rootContainer *container.Container) (map[string]string, error) {
	s := rootContainer.Sandbox
	labels := make(map[string]string, 4)
	labels[prometheus.SandboxIDLabel] = s.ID

	// Compute iteration ID label in a stable manner.
	// This uses sha256(ID + ":" + creation time).
	h := sha256.New()
	if _, err := io.WriteString(h, s.ID); err != nil {
		return nil, err
	}
	if _, err := io.WriteString(h, ":"); err != nil {
		return nil, err
	}
	if _, err := io.WriteString(h, rootContainer.CreatedAt.UTC().String()); err != nil {
		return nil, err
	}
	labels[prometheus.IterationIDLabel] = strconv.FormatUint(binary.BigEndian.Uint64(h.Sum(nil)[:8]), 36)

	if s.PodName != "" {
		labels[prometheus.PodNameLabel] = s.PodName
	}
	if s.Namespace != "" {
		labels[prometheus.NamespaceLabel] = s.Namespace
	}
	return labels, nil
}

// ComputeSpecMetadata returns the labels for the `spec_metadata` metric.
// It merges data from the Specs of multiple containers running within the
// same sandbox.
// This function must support being called with `allContainers` being nil.
// It must return the same set of label keys regardless of how many containers
// are in `allContainers`.
func ComputeSpecMetadata(allContainers []*container.Container) map[string]string {
	const (
		unknownOCIVersion      = "UNKNOWN"
		inconsistentOCIVersion = "INCONSISTENT"
	)

	hasUID0Container := false
	ociVersion := unknownOCIVersion
	for _, cont := range allContainers {
		if cont.RunsAsUID0() {
			hasUID0Container = true
		}
		if ociVersion == unknownOCIVersion {
			ociVersion = cont.Spec.Version
		} else if ociVersion != cont.Spec.Version {
			ociVersion = inconsistentOCIVersion
		}
	}
	return map[string]string{
		"hasuid0":    strconv.FormatBool(hasUID0Container),
		"ociversion": ociVersion,
	}
}
