/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2004-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 */

#include "verify_x509.h"
#include "_verify_x509.h"
#include "log.h"


int verify_X509_init (internal_verify_x509_data_t ** verify_x509_data)
{
    OpenSSL_add_all_algorithms();
    ERR_load_crypto_strings();

    *verify_x509_data = calloc (1, sizeof (internal_verify_x509_data_t));
    (*verify_x509_data)->is_initialized = 1;

    return *verify_x509_data ? 0 : 1;
}

int verify_X509_setParameter (internal_verify_x509_data_t ** verify_x509_data,  int verify_x509_option, ...)
{
    internal_verify_x509_data_t * internal_verify_x509_data;
    struct stat       my_stat;
    unsigned long     result = 0;
    va_list           ap;
    int               retval = ERR_VERIFY_X509_PARAMS_OK; /* all ok */

    if (!verify_x509_data || !(*verify_x509_data))
        return ERR_VERIFY_X509_PARAMS_CONTAINER_FAILURE;
    else
        internal_verify_x509_data = *verify_x509_data;

    va_start (ap, verify_x509_option);
    switch (verify_x509_option) {
        case VERIFY_X509_CA_PATH :
            if (internal_verify_x509_data->capath) {
                retval = ERR_VERIFY_X509_PARAMS_ALREADY_SET;
                goto finalize;
            }
            internal_verify_x509_data->capath = va_arg(ap, char *);

            /* Check of not NULL */
            if (!internal_verify_x509_data->capath) {
                retval = ERR_VERIFY_X509_PARAMS_DATA_EMPTY;
                goto finalize;
            }
            /* Check if we can access the data */
            if ((result = stat(internal_verify_x509_data->capath, &my_stat)) != 0) {
                Error ("Error: unable to stat() CA directory:", "%s",
                       internal_verify_x509_data->capath);
                internal_verify_x509_data->capath = NULL;
                retval = ERR_VERIFY_X509_PARAMS_ACCESS_FAILURE;
                goto finalize;
            }
            break;
        case VERIFY_X509_CERTIFICATE_FILEPATH :
            if (internal_verify_x509_data -> certificate_filepath) {
                retval = ERR_VERIFY_X509_PARAMS_ALREADY_SET;
                goto finalize;
            }
            internal_verify_x509_data->certificate_filepath = va_arg(ap, char *);

            /* Check of not NULL */
            if (!internal_verify_x509_data->certificate_filepath) {
                retval = ERR_VERIFY_X509_PARAMS_DATA_EMPTY;
                goto finalize;
            }
            /* Check if we can access the data */
            if ((result = stat(internal_verify_x509_data->certificate_filepath, &my_stat)) != 0) {
                Error ("Error: unable to stat() Certificate File:", "%s",
                       internal_verify_x509_data->certificate_filepath);
                internal_verify_x509_data->certificate_filepath = NULL;
                retval = ERR_VERIFY_X509_PARAMS_ACCESS_FAILURE;
                goto finalize;
            }
            break;
        case VERIFY_X509_CERTIFICATE_F_HANDLE :
            if (internal_verify_x509_data->certificate_f_handle) {
                retval = ERR_VERIFY_X509_PARAMS_ALREADY_SET;
                goto finalize;
            }
            internal_verify_x509_data->certificate_f_handle = va_arg(ap, FILE *);
            break;
        case VERIFY_X509_CERTIFICATE_PEM :
            if (internal_verify_x509_data->certificate_pem_str) {
                retval = ERR_VERIFY_X509_PARAMS_ALREADY_SET;
                goto finalize;
            }
            internal_verify_x509_data->certificate_pem_str = va_arg(ap, char *);
            break;
        case VERIFY_X509_PRIVATEKEY_FILE :
            if (internal_verify_x509_data->private_key_filepath) {
                retval = ERR_VERIFY_X509_PARAMS_ALREADY_SET;
                goto finalize;
            }
            internal_verify_x509_data->private_key_filepath = va_arg(ap, char *);

            /* Check of not NULL */
            if (!internal_verify_x509_data->private_key_filepath) {
                retval = ERR_VERIFY_X509_PARAMS_DATA_EMPTY;
                goto finalize;
            }
            /* Check if we can access the data */
            if ((result = stat(internal_verify_x509_data->private_key_filepath, &my_stat)) != 0) {
                Error ("Error: unable to stat() Private Key File:", "%s",
                       internal_verify_x509_data->private_key_filepath);
                internal_verify_x509_data->private_key_filepath = NULL;
                retval = ERR_VERIFY_X509_PARAMS_ACCESS_FAILURE;
                goto finalize;
            }
            break;
        case VERIFY_X509_PRIVATEKEY_PEM :
            if (internal_verify_x509_data->private_key_pem) {
                retval = ERR_VERIFY_X509_PARAMS_ALREADY_SET;
                goto finalize;
            }
            internal_verify_x509_data->private_key_pem = va_arg(ap, char *);
            break;
        case VERIFY_X509_CRL_PATH :
            if (internal_verify_x509_data->crl_path) {
                retval = ERR_VERIFY_X509_PARAMS_ALREADY_SET;
                goto finalize;
            }
            internal_verify_x509_data->crl_path = va_arg(ap, char *);

            /* Check of not NULL */
            if (!internal_verify_x509_data->crl_path) {
                retval = ERR_VERIFY_X509_PARAMS_DATA_EMPTY;
                goto finalize;
            }
            /* Check if we can access the data */
            if ((result = stat(internal_verify_x509_data->crl_path, &my_stat)) != 0) {
                Error ("Error: unable to stat() CRL path:", "%s",
                       internal_verify_x509_data->crl_path);
                internal_verify_x509_data->crl_path = NULL;
                retval = ERR_VERIFY_X509_PARAMS_ACCESS_FAILURE;
                goto finalize;
            }
            break;
        case VERIFY_X509_OCSP_RESPONDER_URI :
            if (internal_verify_x509_data->ocsp_responder_uri) {
                retval = ERR_VERIFY_X509_PARAMS_ALREADY_SET;
                goto finalize;
            }
            internal_verify_x509_data->ocsp_responder_uri = va_arg(ap, char *);
            break;
        case VERIFY_X509_OPTIONS_NO_CRL_CHECK :
            internal_verify_x509_data->no_crl_check           = va_arg(ap, unsigned int);
            break;
        case VERIFY_X509_OPTIONS_ALLOW_LIMITED_PROXY :
            internal_verify_x509_data->allow_limited_proxy    = va_arg(ap, unsigned int);
            break;
        case VERIFY_X509_OPTIONS_REQUIRE_LIMITED_PROXY :
            internal_verify_x509_data->require_limited_proxy  = va_arg(ap, unsigned int);
            break;
        case VERIFY_X509_OPTIONS_MUST_HAVE_PRIV_KEY:
            internal_verify_x509_data->must_have_priv_key     = va_arg(ap, unsigned int);
            break;
        case VERIFY_X509_STACK_OF_X509 :
            if (internal_verify_x509_data -> stack_of_x509) {
                retval = ERR_VERIFY_X509_PARAMS_ALREADY_SET;
                goto finalize;
            }
            internal_verify_x509_data->stack_of_x509 = va_arg(ap, STACK_OF (X509) *);
            break;
        case VERIFY_X509_EVP_PKEY :
            if (internal_verify_x509_data->evp_pkey) {
                retval = ERR_VERIFY_X509_PARAMS_ALREADY_SET;
                goto finalize;
            }
            internal_verify_x509_data->evp_pkey = va_arg(ap, EVP_PKEY *);
            break;
        default :
            Error ("Unsupported datatype option specified", "%s\n",
                   "the datatype and data specified is not supported and will not be used in the process");
            retval = ERR_VERIFY_X509_PARAMS_UNSUPPORTED_DATATYPE;
            goto finalize;
    }
finalize:
    va_end(ap);
    return retval;
}


/* Process internal verification data to get to the fundanmental types e.g. STACK_OF (X509) * and EVP_PKEY * */
unsigned long process_internal_verify_data (internal_verify_x509_data_t ** verify_x509_data)
{
    internal_verify_x509_data_t * internal_verify_x509_data = NULL;
    unsigned long result = X509_V_ERR_APPLICATION_VERIFICATION;

    /* Input check */
    if (!verify_x509_data || !(*verify_x509_data)) {
        return X509_V_ERR_APPLICATION_VERIFICATION;
    }
    internal_verify_x509_data = *verify_x509_data;

    /* If the private key is already set, don't look for it */
    if (!(internal_verify_x509_data->evp_pkey))
    {
        /* To EVP Private Key from Source Private Key PEM String */
        if (internal_verify_x509_data->private_key_pem) {
            result = verify_x509_readPrivateKeyFromPEM(internal_verify_x509_data->private_key_pem,
                                                       &(internal_verify_x509_data->derived_data.evp_pkey));
            if (X509_V_OK != result) {
                Error ("Failed to read the private key in file:", "%s\n",
                        internal_verify_x509_data -> certificate_filepath);
                return result;
            }
        }

        /* To EVP Private Key from Source Certificate PEM String */
        if (internal_verify_x509_data->certificate_pem_str) {
            result = verify_x509_readPrivateKeyFromPEM(internal_verify_x509_data->certificate_pem_str,
                                                       &(internal_verify_x509_data->derived_data.evp_pkey));
            if (X509_V_OK != result) {
                Error ("Failed to read the private key in file:", "%s\n",
                        internal_verify_x509_data->certificate_filepath);
                return result;
            }
        }

        /* To EVP Private Key from Source Private Key File at path */
        else if (internal_verify_x509_data->private_key_filepath) {
            result = verify_x509_readPrivateKeyFromFile(internal_verify_x509_data->private_key_filepath,
                                                        &(internal_verify_x509_data->derived_data.evp_pkey));
            if (X509_V_OK != result) {
                Error ("Failed to read the private key in file:", "%s\n",
                        internal_verify_x509_data -> certificate_filepath);
                return result;
            }
        }

        /* To EVP Private Key from Source Certificate File at path */
        else if (internal_verify_x509_data->certificate_filepath) {
            result = verify_x509_readPrivateKeyFromFile(internal_verify_x509_data->certificate_filepath,
                                                        &(internal_verify_x509_data->derived_data.evp_pkey));
            if (X509_V_OK != result) {
                Error ("Failed to read the private key in file:", "%s\n",
                        internal_verify_x509_data -> certificate_filepath);
                return result;
            }
        }
    }

    /* If the certificate stack is already provided, don't go look for it in the certificate file */
    if (!(internal_verify_x509_data->stack_of_x509)) {
        result = verify_x509_readPublicCertChain(internal_verify_x509_data->certificate_filepath,
                                                 &(internal_verify_x509_data->derived_data.stack_of_x509));
        if (X509_V_OK != result) {
            Error ("Failed to read the certificate stack in file:", "%s\n",
                    internal_verify_x509_data -> certificate_filepath);
            return result;
        }
    }

    return X509_V_OK;
}



/******************************************************************************
Function:
    verify_X509_verify

Description:
    This is the verification function that will first process the provided
    setParameters into usable datatypes for the actual verification. Then the
    verification of the certificate chain (includes GT2 and GT4/RFC proxies)
    will start. First the chain will be checked and then (if there is a private
    key) the private key must match the certificate chain.

Parameters:
    internal_verify_x509_data_t ** verify_x509_data

Return
    X509_V_OK  :  all is perfect
    !X509_V_OK :  error code
******************************************************************************/
int verify_X509_verify (internal_verify_x509_data_t **verify_x509_data)
{
    internal_verify_x509_data_t * internal_verify_x509_data = NULL;
    unsigned long result = X509_V_ERR_APPLICATION_VERIFICATION;
    int i = 0;
    int depth = 0;
    X509 *cert = NULL;
    lcmaps_proxy_type_t p_type = NONE;

    /* Input check */
    if (!verify_x509_data || !(*verify_x509_data)) {
        return X509_V_ERR_APPLICATION_VERIFICATION;
    }
    internal_verify_x509_data = *verify_x509_data;


    /*
     * Crunch on the supplied information to get to the principle pieces of information required
     * for the verification based on the STACK_OF (X509) * and the EVP_PKEY *
     */
    result = process_internal_verify_data(&internal_verify_x509_data);
    if (X509_V_OK != result) {
        return result;
    }

    /* first, verify the certificate chain */
    if (!internal_verify_x509_data->stack_of_x509 &&
        !internal_verify_x509_data->derived_data.stack_of_x509) {
        Error("No certificate chain present",
              "There was no STACK_OF(X509) provided, nor a PEM string to be transcoded into a STACK_OF(X509)\n");
        return X509_V_ERR_APPLICATION_VERIFICATION;
    }

    /* Verify the certificate chain and its content */
    result = grid_verifyCert (internal_verify_x509_data -> capath,
                              internal_verify_x509_data -> stack_of_x509 ?
                                    internal_verify_x509_data -> stack_of_x509 :
                                    internal_verify_x509_data -> derived_data.stack_of_x509);
    if (result != X509_V_OK) {
        Error( "Verifying certificate chain", "%s\n",
               X509_verify_cert_error_string(result));
        return result;
    }

#if 1
/* Twee verschillende Seg faults */
    /*
     * When set to have the private key, then this means that the certificate chain
     * MUST be accompanied by a private key (and in other test this one must fit
     * the chain)
     */
    if ((internal_verify_x509_data->must_have_priv_key == VERIFY_ENABLE) &&
        !internal_verify_x509_data->evp_pkey &&
        !internal_verify_x509_data->derived_data.evp_pkey) {
        Error ("No private key provided",
               "the configuration (by default or by explict statement) demands its presence\n");
        return VERIFY_X509_OPTIONS_MUST_HAVE_PRIV_KEY;
    } else if (!internal_verify_x509_data->evp_pkey &&
               !internal_verify_x509_data->derived_data.evp_pkey) {
        Log (L_INFO, "Verification of chain without private key is OK\n");
        result = X509_V_OK; /* Good */
    } else {
        /* still OK? then match the proxy public and private keys */
        result = grid_verifyPrivateKey (internal_verify_x509_data -> stack_of_x509 ?
                                            internal_verify_x509_data -> stack_of_x509 :
                                            internal_verify_x509_data -> derived_data.stack_of_x509,
                                        internal_verify_x509_data -> evp_pkey ?
                                            internal_verify_x509_data -> evp_pkey :
                                            internal_verify_x509_data -> derived_data.evp_pkey);
        if (result != X509_V_OK ) {
            Error( "Verifying private key", "%s\n",
                    ERR_reason_error_string(result));
            return result;
        }
        /* aaah, nirwana */
        Log (L_INFO, "Verification of chain with private key is OK\n");
    }
#endif

    /* Test for Limited proxies in the certificate stack, and fail when found */
    if (internal_verify_x509_data->allow_limited_proxy == VERIFY_DISABLE) {
        depth = sk_X509_num(internal_verify_x509_data -> stack_of_x509 ?
                                internal_verify_x509_data -> stack_of_x509 :
                                internal_verify_x509_data -> derived_data.stack_of_x509);

        for (i = 0; i < depth; i++) {
            cert = sk_X509_value(internal_verify_x509_data -> stack_of_x509 ?
                                     internal_verify_x509_data -> stack_of_x509 :
                                     internal_verify_x509_data -> derived_data.stack_of_x509, i);
            if (cert != NULL) {
                p_type = lcmaps_type_of_proxy(cert);
                if ((p_type == RFC_LIMITED_PROXY) || (p_type == GT2_LIMITED_PROXY) || (p_type == GT3_LIMITED_PROXY)) {
                    Error( "Checking for limited proxy usage", "Found a limited proxy in the certificate chain but this is disallowed by configuration.\n");
                    return X509_V_ERR_APPLICATION_VERIFICATION;
                }
            }
        }
    }

    return result;
}


int verify_X509_term (internal_verify_x509_data_t **verify_x509_data)
{
    internal_verify_x509_data_t *internal_verify_x509_data = NULL;

    if (!verify_x509_data || !(*verify_x509_data)) {
        return 1;
    }
    internal_verify_x509_data = *verify_x509_data;

    /* if nothing initialized, or already cleaned, nothing to do */
    if (!internal_verify_x509_data->is_initialized) {
        return 0;
    }

    /* Liberate derived/acquired/extracted information */
    if (internal_verify_x509_data->derived_data.certificate_pem_str) {
        free(internal_verify_x509_data->derived_data.certificate_pem_str);
        internal_verify_x509_data->derived_data.certificate_pem_str = NULL;
    }
    if (internal_verify_x509_data->derived_data.private_key_pem) {
        free(internal_verify_x509_data->derived_data.private_key_pem);
        internal_verify_x509_data->derived_data.private_key_pem = NULL;
    }
    if (internal_verify_x509_data->derived_data.ocsp_responder_uri) {
        free(internal_verify_x509_data->derived_data.ocsp_responder_uri);
        internal_verify_x509_data->derived_data.ocsp_responder_uri = NULL;
    }
    if (internal_verify_x509_data->derived_data.stack_of_x509) {
        /* popper-de-pop */
        sk_X509_pop_free(internal_verify_x509_data->derived_data.stack_of_x509, X509_free);
        internal_verify_x509_data->derived_data.stack_of_x509 = NULL;
    }
    if (internal_verify_x509_data->derived_data.evp_pkey) {
        EVP_PKEY_free(internal_verify_x509_data->derived_data.evp_pkey);
        internal_verify_x509_data->derived_data.evp_pkey = NULL;
    }

    /* Nullify memory */
    memset(internal_verify_x509_data, 0, sizeof(internal_verify_x509_data_t));

    /* Free and clean the struct */
    free(internal_verify_x509_data);
    internal_verify_x509_data = NULL;
    *verify_x509_data         = NULL;

    return 0;
}
