#if   !defined(__MEMORYSTORAGE_HPP)
#define  __MEMORYSTORAGE_HPP

/*
  CoreLinux++ 
  Copyright (C) 1999,2000 CoreLinux Consortium
  
   The CoreLinux++ Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The CoreLinux++ Library Library is distributed in the hope that it will 
   be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  
*/

#if   !defined(__COMMON_HPP)
#include <Common.hpp>
#endif

#if   !defined(__TRANSIENTSTORAGE_HPP)
#include <TransientStorage.hpp>
#endif

#if   !defined(__BOUNDSEXCEPTION_HPP)
#include <BoundsException.hpp>
#endif

namespace corelinux
{
   DECLARE_CLASS( MemoryStorage );

   /**
   MemoryStorage is type of Storage characterized as transient
   and high speed.
   The interface provides pointer semantics with the addition
   of bounds error checking.
   */

   class MemoryStorage : public TransientStorage, Synchronized
   {

   public:


      //
      // Constructors and destructor
      //
                        /**
                        Constructor for storage object
                        @param MemoryIdentifier a unique id
                        @param Int size of region in bytes
                        @param void * to base address of region
                        */

                        MemoryStorage( MemoryIdentifierCref , IntCref , VoidPtr );

      //
      // Operator overloads
      //

                     /// Compares identifier

               bool  operator==( MemoryStorageCref ) const;

                     /// Returns identifier

                     operator MemoryIdentifierCref( void ) const;

      //
      // Accessors
      //

                     /// Increment current pointer

                     void  operator+( Int ) throw( BoundsException );

                     /// Decrement current pointer

                     void  operator-( Int ) throw( BoundsException );

                     /// Reads current location as type

                     template< class T > 
                        operator T( void ) throw( BoundsException )
                     {
                        // Bounds check first

                        if( Dword( ( BytePtr(theCurrentPointer) + sizeof(T) ) - 
                                  BytePtr(theBasePointer)) > Dword(theSize) )
                        {
                           throw BoundsException( LOCATION );
                        }
                        else
                        {
                           ;  // do nothing
                        }

                        return ((T *)theCurrentPointer)[0];
                     }

                     /// Reads current location as type pointer

                     template< class T > 
                        operator T*( void ) throw( BoundsException )
                     {
                        // Bounds check first

                        if( Dword( ( BytePtr(theCurrentPointer) + sizeof(T) ) - 
                                  BytePtr(theBasePointer)) > Dword(theSize) )
                        {
                           throw BoundsException( LOCATION );
                        }
                        else
                        {
                           ;  // do nothing
                        }

                        return ((T *)theCurrentPointer);
                     }
      //
      // Mutators
      //
                     /// Data assignment

                     template< class T > 
                        T & operator=( T & aT ) throw( BoundsException )
                     {
                        if( Dword( ( BytePtr(theCurrentPointer) + sizeof(T) ) - 
                                  BytePtr(theBasePointer)) > Dword(theSize) )
                        {
                           throw BoundsException( LOCATION );
                        }
                        else
                        {
                           ;  // do nothing
                        }

                        GUARD;
                        std::memmove( theCurrentPointer, &aT, sizeof(T) );

                        return aT;
                     }

                     /// Subscript offset operator

                     MemoryStorageRef  operator[]( Int offset ) 
                           throw( BoundsException );
      //
      // Traversal operations
      //

                     /**
                     forEach invokes the callers method to perform whatever 
                     operation they want on the type reference in the segment
                     space.
                     @param Xexec call to templated type
                     */

                     template < class Type, class Xexec >
                        void  forEach( Xexec aExec ) throw ( Assertion )
                     {
                        REQUIRE( aExec != NULLPTR );

                        GUARD;

                        Type  *pType( (Type *)theBasePointer );
                        Int   maxCount( theSize / sizeof( Type ) );

                        for( Int x = 0; x < maxCount; ++x, ++pType )
                        {
                           (*aExec)(pType);
                        }
                     }
                     

                     /**
                     forEach that invokes the callers method if the
                     callers test method returns true
                     @param Xexec call to templated type
                     @param Test call to templated type
                     */

                     template < class Type, class Xexec, class Test >
                        void  forEach( Xexec aExec, Test aTest ) 
                           throw ( Assertion )
                     {
                        REQUIRE( aExec != NULLPTR );
                        REQUIRE( aTest != NULLPTR );

                        GUARD;

                        Type  *pType( (Type *)theBasePointer );
                        Int   maxCount( theSize / sizeof( Type ) );

                        for( Int x = 0; x < maxCount; ++x, ++pType )
                        {
                           if( (*aTest)(pType) == true )
                           {
                              (*aExec)(pType);
                           }
                           else
                           {
                              ;  // do nothing
                           }
                        }
                     }


   protected:


               friend class Memory;

                        /// Default is never called

                        MemoryStorage( void ) throw( Assertion );
                  
                        /// Copy called by Memory 

                        MemoryStorage( MemoryStorageCref );

                        /// Virtual destructor

      virtual           ~MemoryStorage( void );

                        /// Assignment called by Memory

               MemoryStorageRef  operator=( MemoryStorageCref );

                        /// Internal reference to base

               VoidPtr  getBasePointer( void );

   private:

               /// Unique identifier

               MemoryIdentifier  theIdentifier;

               /// The size of region in bytes

               Int               theSize;

               /// The base pointer of the region

               VoidPtr           theBasePointer;

               /// The current pointer into the region

               VoidPtr           theCurrentPointer;
   };

}

#endif // if !defined(__MEMORYSTORAGE_HPP)

/*
   Common rcs information do not modify
   $Author: frankc $
   $Revision: 1.4 $
   $Date: 2000/06/10 01:32:17 $
   $Locker:  $
*/

