"""Wrappers around ARC client commands and related utilities."""

import os
import posix
import re
import sys
import time
from subprocess import Popen, PIPE, CalledProcessError
from arcnagios.utils import map_option, host_of_uri, ResultOk, ResultError

class ParseError(Exception):
    """Exception raised on unrecognized command output."""
    pass

# Job States
#

S_UNKNOWN = 0
S_ENTRY = 1
S_PRERUN = 2
S_INLRMS = 3
S_POSTRUN = 4
S_FINAL = 5

class Jobstate(object):
    def __init__(self, name, stage):
        self.name = name
        self.stage = stage
    def __str__(self):
        return self.name
    def is_final(self):
        return self.stage == S_FINAL

class InlrmsJobstate(Jobstate):
    def __init__(self, name):
        Jobstate.__init__(self, name, S_INLRMS)

class PendingJobstate(Jobstate):
    def __init__(self, name, stage, pending):
        Jobstate.__init__(self, name, stage)
        self.pending = pending

def _jobstate(name, stage = S_UNKNOWN):
    if name.startswith('INLRMS:'):
        assert stage == S_UNKNOWN or stage == S_INLRMS
        js = InlrmsJobstate(name)
    elif name.startswith('PENDING:'):
        pending = jobstate_of_str(name[8:])
        js = PendingJobstate(name, stage, pending)
    else:
        js = Jobstate(name, stage)
    _jobstate_of_str[name] = js
    return js

_jobstate_of_str = {}
def jobstate_of_str(name):
    if not name in _jobstate_of_str:
        _jobstate_of_str[name] = _jobstate(name)
    return _jobstate_of_str[name]

J_NOT_SEEN      = _jobstate("NOT_SEEN",         stage = S_ENTRY)
J_ACCEPTED      = _jobstate("Accepted",         stage = S_ENTRY)
J_PREPARING     = _jobstate("Preparing",        stage = S_PRERUN)
J_SUBMITTING    = _jobstate("Submitting",       stage = S_PRERUN)
J_HOLD          = _jobstate("Hold",             stage = S_PRERUN)
J_QUEUING       = _jobstate("Queuing",          stage = S_INLRMS)
J_RUNNING       = _jobstate("Running",          stage = S_INLRMS)
J_FINISHING     = _jobstate("Finishing",        stage = S_POSTRUN)
J_FINISHED      = _jobstate("Finished",         stage = S_FINAL)
J_KILLED        = _jobstate("Killed",           stage = S_FINAL)
J_FAILED        = _jobstate("Failed",           stage = S_FINAL)
J_DELETED       = _jobstate("Deleted",          stage = S_FINAL)
J_UNDEFINED     = _jobstate("Undefined",        stage = S_UNKNOWN)
J_OTHER         = _jobstate("Other",            stage = S_UNKNOWN)

# Also added to _jobstate_of_str for backwards compatibility.  We used
# "Specific state" earlier, and it may be saved in active.map files.
SJ_ACCEPTING            = _jobstate("ACCEPTING",        stage = S_ENTRY)
SJ_PENDING_ACCEPTED     = _jobstate("PENDING:ACCEPTED", stage = S_ENTRY)
SJ_ACCEPTED             = _jobstate("ACCEPTED",         stage = S_ENTRY)
SJ_PENDING_PREPARING    = _jobstate("PENDING:PREPARING",stage = S_PRERUN)
SJ_PREPARING            = _jobstate("PREPARING",        stage = S_PRERUN)
SJ_SUBMIT               = _jobstate("SUBMIT",           stage = S_PRERUN)
SJ_SUBMITTING           = _jobstate("SUBMITTING",       stage = S_PRERUN)
SJ_PENDING_INLRMS       = _jobstate("PENDING:INLRMS",   stage = S_INLRMS)
SJ_INLRMS               = _jobstate("INLRMS",           stage = S_INLRMS)
SJ_INLRMS_Q             = _jobstate("INLRMS:Q",         stage = S_INLRMS)
SJ_INLRMS_R             = _jobstate("INLRMS:R",         stage = S_INLRMS)
SJ_INLRMS_EXECUTED      = _jobstate("INLRMS:EXECUTED",  stage = S_INLRMS)
SJ_INLRMS_S             = _jobstate("INLRMS:S",         stage = S_INLRMS)
SJ_INLRMS_E             = _jobstate("INLRMS:E",         stage = S_INLRMS)
SJ_INLRMS_O             = _jobstate("INLRMS:O",         stage = S_INLRMS)
SJ_FINISHING            = _jobstate("FINISHING",        stage = S_POSTRUN)
SJ_CANCELING            = _jobstate("CANCELING",        stage = S_POSTRUN)
SJ_FAILED               = _jobstate("FAILED",           stage = S_FINAL)
SJ_KILLED               = _jobstate("KILLED",           stage = S_FINAL)
SJ_FINISHED             = _jobstate("FINISHED",         stage = S_FINAL)
SJ_DELETED              = _jobstate("DELETED",          stage = S_FINAL)



# Utilities
#

def explain_wait_status(status):
    if posix.WIFEXITED(status):
        exitcode = posix.WEXITSTATUS(status)
        if exitcode:
            msg = 'exited with %d'%exitcode
        else:
            msg = 'exited normally'
    elif posix.WIFSTOPPED(status):
        signo = posix.WSTOPSIG(status)
        msg = 'stopped by signal %d'%signo
    elif posix.WIFSIGNALED(status):
        signo = posix.WTERMSIG(status)
        msg = 'terminated by signal %d'%signo
    elif posix.WIFCONTINUED(status):
        msg = 'continued'
    if posix.WCOREDUMP(status):
        return '%s (core dumped)' % msg
    else:
        return msg

def roundarg(t):
    return str(int(t + 0.5))


# ARC Commands
#

_arcstat_state_re = re.compile(r'(\w+)\s+\(([^()]+)\)')
def parse_old_arcstat_state(s):
    mo = re.match(_arcstat_state_re, s)
    if not mo:
        raise ParseError('Malformed arcstat state %s.'%s)
    return jobstate_of_str(mo.group(1)), mo.group(2)

class Arcstat(object):
    def __init__(self,
                 state = None, specific_state = None,
                 submitted = None,
                 job_error = None,
                 exit_code = None):
        self.state = state
        self.specific_state = specific_state
        self.submitted = submitted
        self.job_error = job_error
        self.exit_code = exit_code

def arcstat(jobids = None, log = None, timeout = 5, show_unavailable = False):
    cmd = ['arcstat', '-l', '--timeout', str(timeout)]
    if jobids is None:
        cmd.append('-a')
    else:
        cmd.extend(map(str, jobids))
    if show_unavailable:
        cmd.append('-u')
    fd = Popen(cmd, stdout = PIPE).stdout
    jobstats = {}
    lnno = 0

    def parse_error(msg):
        if log:
            log.error('Unexpected output from arcstat '
                      'at line %d: %s'%(lnno, msg))
        else:
            fd.close()
            raise ParseError('Unexpected output from arcstat '
                             'at line %d: %s'%(lnno, msg))
    def convert(jobid, jobstat):
        if jobstat['State'] == 'Undefined':
            state = J_UNDEFINED
            specific_state = None
        elif 'Specific state' in jobstat:
            state = jobstate_of_str(jobstat['State'])
            specific_state = jobstat['Specific state']
        elif 'State' in jobstat:
            # Compatibility with old arcstat.
            state, specific_state = parse_old_arcstat_state(jobstat['State'])
        else:
            raise ParseError('Missing "State" or "Specific state" for %s.'
                             % jobid)
        return Arcstat(state = state, specific_state = specific_state,
                       exit_code = map_option(int, jobstat.get('Exit code')),
                       submitted = jobstat.get('Submitted'),
                       job_error = jobstat.get('Job Error'))

    jobid, jobstat, jobfield = None, {}, None
    for ln_enc in fd:
        ln = ln_enc.decode('utf-8')
        lnno += 1
        if ln.endswith('\n'): ln = ln[0:-1]

        if ln.startswith('No jobs') or ln.startswith('Status of '):
            break
        elif ln.startswith('Job:'):
            if not jobid is None:
                jobstats[jobid] = convert(jobid, jobstat)
            jobid = ln[4:].strip()
            jobstat = {}
            jobfield = None
        elif ln.startswith('Warning:'):
            if log:
                log.warning(ln)
        elif ln == '':
            pass
        elif ln.startswith('  '):
            if jobfield is None:
                parse_error('Continuation line %r before job field.')
                continue
            else:
                jobstat[jobfield] += '\n' + ln
        elif ln.startswith(' '):
            kv = ln.strip()
            try:
                jobfield, v = kv.split(':', 1)
                if jobid is None:
                    parse_error('Missing "Job: ..." header before %r' % kv)
                    continue
                else:
                    jobstat[jobfield] = v.strip()
            except ValueError:
                parse_error('Expecting "<key>: <value>", got %r' % ln)
                continue
        else:
            parse_error('Unrecognized output %r' % ln)

    fd.close()
    if not jobid is None:
        jobstats[jobid] = convert(jobid, jobstat)
    return jobstats

class ArclsEntry(object):
    DIR = 0
    FILE = 1

    def __init__(self, name, typ, size, cdate, ctod, validity, checksum, latency = ''):
        self.filename = name
        if typ == 'dir':
            self.entry_type = self.DIR
        elif typ == 'file':
            self.entry_type = self.FILE
        else:
            self.entry_type = None
        self.size = size
#       Not in Python 2.4:
#       self.ctime = datetime.strptime(cdate + 'T' + ctod, '%Y-%m-%dT%H:%M:%S')
        def drop_NA(s):
            if s != '(n/a)':
                return s
            else:
                return None

        if cdate == '(n/a)':
            self.validity = drop_NA(validity)
            self.checksum = drop_NA(checksum)
            self.latency = drop_NA(latency)
        else:
            self.validity = drop_NA(ctod)
            self.checksum = drop_NA(validity)
            self.latency = drop_NA(checksum)

class PerfProcess(object):
    _bindir = ''
    program = None

    def __init__(self, args, perflog, perfindex):
        program_path = os.path.join(self._bindir, self.program)
        self._command = [program_path] + list(map(str, args))
        self._perflog = perflog
        self._perfindex = perfindex
        self._start_time = time.time()
        self._popen = Popen(self._command, stdout = PIPE, stderr = PIPE)
        self._result = None

    def _ok(self, stdout):
        return ResultOk(stdout)

    def _error(self, returncode, stderr):
        exn = CalledProcessError(returncode, self._command, stderr)
        return ResultError(exn)

    def _process_result(self):
        stdout, stderr = self._popen.communicate()
        returncode = self._popen.returncode
        run_time = time.time() - self._start_time
        if self._perflog:
            self._perflog.addi(self.program + '_time', self._perfindex,
                               run_time, uom = 's', limit_min = 0)
        if returncode == 0:
            self._result = self._ok(stdout.decode('utf-8'))
        else:
            self._result = self._error(returncode, stderr.decode('utf-8'))

    def communicate(self):
        if not self._result:
            self._process_result()
        return self._result

    def poll(self):
        poll_result = self._popen.poll()
        if not poll_result is None:
            self._process_result()
        return poll_result

class ArcsubProcess(PerfProcess):
    program = 'arcsub'

    def __init__(self, jobdesc_files, cluster = None, jobids_to_file = None,
                 timeout = None, perflog = None):
        args = jobdesc_files
        if cluster:
            args += ['-c', cluster]
            if ':' in cluster:
                perfindex = host_of_uri(cluster)
            else:
                perfindex = cluster
        else:
            perfindex = None
        if jobids_to_file: args += ['-o', jobids_to_file]
        if timeout: args += ['-t', timeout]
        PerfProcess.__init__(self, args, perflog, perfindex)

class ArcgetProcess(PerfProcess):
    program = 'arcget'

    def __init__(self, job_id, top_output_dir = None,
                 timeout = None, perflog = None):
        ce_host = host_of_uri(job_id)
        args = [job_id]
        if not timeout is None: args += ['-t', roundarg(timeout)]
        if not top_output_dir is None: args += ['-D', top_output_dir]
        PerfProcess.__init__(self, args, perflog, ce_host)

class ArckillProcess(PerfProcess):
    program = 'arckill'

    def __init__(self, job_id, force = False, timeout = None, perflog = None):
        # pylint: disable=unused-argument
        ce_host = host_of_uri(job_id)
        args = [job_id]
        if not timeout is None: args += ['-t', roundarg(timeout)]
        PerfProcess.__init__(self, args, perflog, ce_host)

class ArccleanProcess(PerfProcess):
    program = 'arcclean'

    def __init__(self, job_id, force = False, timeout = None, perflog = None):
        ce_host = host_of_uri(job_id)
        args = [job_id]
        if not timeout is None: args += ['-t', roundarg(timeout)]
        if force: args.append('-f')
        PerfProcess.__init__(self, args, perflog, ce_host)

class ArcrmProcess(PerfProcess):
    program = 'arcrm'

    def __init__(self, url, force = False, timeout = None, perflog = None):
        se_host = host_of_uri(url)
        args = [url]
        if not timeout is None: args += ['-t', roundarg(timeout)]
        if force: args.append('-f')
        PerfProcess.__init__(self, args, perflog, se_host)

class ArccpProcess(PerfProcess):
    program = 'arccp'

    def __init__(self, src_url, dst_url, timeout = 20, transfer = True,
                 perflog = None):
        se_host = None
        if   ':' in src_url: se_host = host_of_uri(src_url)
        elif ':' in dst_url: se_host = host_of_uri(dst_url)
        args = [src_url, dst_url]
        if not timeout is None: args += ['-t', roundarg(timeout)]
        if not transfer: args.append('-T')
        PerfProcess.__init__(self, args, perflog, se_host)

class ArclsProcess(PerfProcess):
    program = 'arcls'

    def __init__(self, url, timeout = 20, perflog = None):
        se_host = host_of_uri(url)
        args = ['-l', url]
        if not timeout is None: args += ['-t', roundarg(timeout)]
        PerfProcess.__init__(self, args, perflog, se_host)

    def _ok(self, stdout):
        entries = []
        for ln in stdout.split('\n')[1:]:
            if not ln:
                continue
            comps = ln.rsplit(None, 6)
            if len(comps) != 7:
                raise RuntimeError('Unexpected line %r from %s'
                                   % (ln, ' '.join(self._command)))
            entries.append(ArclsEntry(*comps))
        return ResultOk(entries)

class ArclslocProcess(PerfProcess):
    program = 'arcls'

    def __init__(self, url, timeout = 20, perflog = None):
        se_host = host_of_uri(url)
        args = ['-L', '-l', url]
        if not timeout is None: args += ['-t', roundarg(timeout)]
        PerfProcess.__init__(self, args, perflog, se_host)

    def _ok(self, stdout):
        return ResultOk([s.strip() for s in stdout.split('\n')[1:]])

class ArcClient(object):

    def __init__(self, perflog = None):
        self._perflog = perflog

    def arcsub(self, *args, **kwargs):
        return ArcsubProcess(*args, perflog = self._perflog, **kwargs).communicate()

    def arcget(self, *args, **kwargs):
        return ArcgetProcess(*args, perflog = self._perflog, **kwargs).communicate()

    def arcrm(self, *args, **kwargs):
        return ArcrmProcess(*args, perflog = self._perflog, **kwargs).communicate()

    def arcclean(self, *args, **kwargs):
        return ArccleanProcess(*args, perflog = self._perflog, **kwargs).communicate()

    def arckill(self, *args, **kwargs):
        return ArckillProcess(*args, perflog = self._perflog, **kwargs).communicate()

    def arccp(self, *args, **kwargs):
        return ArccpProcess(*args, perflog = self._perflog, **kwargs).communicate()

    def arcls(self, *args, **kwargs):
        return ArclsProcess(*args, perflog = self._perflog, **kwargs).communicate()

    def arcls_L(self, *args, **kwargs):
        return ArclslocProcess(*args, perflog = self._perflog, **kwargs).communicate()
