/*
 * Copyright 2008 Klaus Triendl
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free 
 * Software Foundation, 51 Franklin Street, Fifth Floor, 
 * Boston, MA 02110-1301, USA.
*/

#include <iostream>
#include <sigc++/adaptors/lambda/base.h>	// sigc::var
#include <sigc++/signal.h>
#include <glibmm/main.h>
#include <glibmm/thread.h>
#include <sigx/glib_auto_dispatchable.h>
#include <sigx/glib_threadable.h>
#include <sigx/signal_f.h>
#include <sigx/connection_wrapper.h>


using namespace std;


// fwd decl
class MainThread;

class MyThread: public sigx::glib_threadable
{
public:
	MyThread();


protected:
	// virtuals from sigx::threadable
	virtual void on_startup();


private:
	struct ThreadData;
	Glib::Private<ThreadData> m_ThreadData;


// signal api
public:
	// expose the 'nothing' signal
	sigx::signal_f<sigc::signal<void> > signal_nothing;
};


struct MyThread::ThreadData
{
	sigc::signal<void> m_sigNothing;
};


class MainThread: public sigx::glib_auto_dispatchable
{
public:
	MainThread();
	void run();

protected:
	void on_self_ready();
	void on_end_mainthread();


protected:
	Glib::RefPtr<Glib::MainLoop> m_mainloop;
	MyThread m_mythread;
	sigx::connection_wrapper m_connMyThreadNothing;
};





using namespace std;


MyThread::MyThread(): 
	sigx::glib_threadable(), 
	// initialize the signal api
	signal_nothing(*this, m_ThreadData, &ThreadData::m_sigNothing)
{}

//virtual 
void MyThread::on_startup()
{
	m_ThreadData.set(new ThreadData);
}




MainThread::MainThread(): 
	sigx::glib_auto_dispatchable(), 
	m_mainloop(), 
	m_mythread(), 
	m_connMyThreadNothing()
{}

void MainThread::run()
{
	m_mainloop = Glib::MainLoop::create();

	// one-shot idle handler
	Glib::signal_idle().connect(
		sigc::bind_return(
			sigc::mem_fun(this, &MainThread::on_self_ready), 
			false
		)
	);
	
	m_mainloop->run();
}

void MainThread::on_end_mainthread()
{
	// disconnecting after restarting mythread shouldn't work (connection invalid) but yield no errors
	m_connMyThreadNothing.disconnect();
	// destroying after restarting mythread shouldn't (connection invalid) work but yield no errors
	m_connMyThreadNothing = sigx::connection_wrapper();
	m_mythread.finish();
	m_mainloop->quit();
}

void do_nothing_dummy()
{}

void MainThread::on_self_ready()
{
	// start mythread
	m_mythread.run();
	// connect to the thread's 'nothing' signal
	m_connMyThreadNothing = 
		m_mythread.signal_nothing().connect(
			sigx::open_tunnel_with(&do_nothing_dummy, *this)
		);
	// wait and test the connection status
	m_connMyThreadNothing.connected();
	// restart the thread
	m_mythread.finish();
	m_mythread.run();

	// one-shot idle handler to 
	Glib::signal_idle().connect(
		sigc::bind_return(
			sigc::mem_fun(this, &MainThread::on_end_mainthread), 
			false
		)
	);
}



int main()
{
	Glib::thread_init();

	MainThread app;
	app.run();

	cout << "press <enter>.";
	cin.get();
	return 0;
}
