use core:io;

/**
 * Describes a PDF file.
 *
 * A PDF file contains a number of objects, each described by an instance of PdfObject. These
 * objects may contain things like postscript code, images etc.
 */
class File {
	// The root object of this file. Add pages to the file through this object.
	Catalog root;

	// Initialize. Creates a Catalog object as the root object.
	init() {
		init() {}

		add(root);
	}

	// Write the pdf structure to file.
	void write(OStream to) {
		MeterOStream meter(to);
		Word[] offsets;

		writeHeader(meter);
		for (o in objects) {
			offsets << meter.tell();
			o.write(meter);
		}

		writeFooter(meter.tell, offsets, meter);
	}

	// Add an object to this file.
	void add(PdfObject o) {
		objects << o;
		o.id = objects.count;
		o.addedTo(this);
	}

private:
	// All objects in this file.
	PdfObject[] objects;


	// Write the PDF header.
	void writeHeader(OStream to) {
		var output = textOutput(to);
		output.write("%PDF-1.6\n%✓\n"); // Check mark to use some high-order bits.
		output.flush();
	}

	// Write the footer.
	void writeFooter(Word pos, Word[] offsets, OStream to) {
		StrBuf out;
		out << fill(Char(48)); // 0
		out << "xref\r\n";
		out << "0 " << (offsets.count + 1) << "\n";

		// First offset is special.
		out << "0000000000 65535 f\n";
		for (o in offsets) {
			out << width(10) << o << " 00000 n\n";
		}

		// Trailer to locate the xref table.
		// TODO: Allow specifying an arbitrary root object?
		out << "trailer <</Size " << (offsets.count + 1) << " /Root 1 0 R>>\n";
		out << "startxref\n" << pos << "\n%%EOF";

		// This section needs CR LF line endings as opposed to the rest of the file.
		TextInfo info;
		info.useCrLf = true;
		info.useBom = false;
		Utf8Output conv(to, info);
		conv.write(out.toS);
		conv.flush();
	}
}

// Create a proper text stream.
package TextOutput textOutput(OStream to) {
	TextInfo info;
	info.useCrLf = false;
	info.useBom = false;
	return Utf8Output(to, info);
}
