/*
 * radio plugin for Xfce4.
 *
 * Copyright (C) 2006-2011 Stefan Ott, All rights reserved.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Plugin layout
 *
 * ,--[ ebox ]--------------------------------------------------.
 * |,--[ outer_box ] ------------------------------------------.|
 * ||                                                          ||
 * || ,--[label]----. ,--[ box ] ------. ,--[ signal_image ]-. ||
 * || |             | |                | |                   | ||
 * || |             | | [signal_bar]   | |                   | ||
 * || |             | |                | |                   | ||
 * || '-------------' '----------------' '-------------------' ||
 * |'----------------------------------------------------------'|
 * '------------------------------------------------------------'
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>

#include "xfce4-radio.h"
#include "../icons/signal.xpm"

#include "radio.h"
#include "v4l2.h"

#include <libxfcegui4/libxfcegui4.h>
#include <gdk/gdkkeysyms.h>

#include <linux/videodev2.h>
#include <dirent.h>

#define SIGNAL_WIDTH 15
#define SIGNAL_HEIGHT 12

static void
update_signal_image(radio_gui* data, gint signal)
{
	GdkImage *image;
	GdkBitmap *mask;
	GdkPixmap *signal_s;

	signal_s = gdk_pixmap_create_from_xpm_d
		(GTK_WIDGET(data->plugin)->window, &mask, NULL, signal_xpm);
	image = gdk_drawable_get_image(signal_s, SIGNAL_WIDTH * signal, 0,
		SIGNAL_WIDTH, SIGNAL_HEIGHT);

	gtk_image_set_from_image(GTK_IMAGE(data->signal_image), image, mask);
	gtk_widget_show(data->signal_image);

	g_object_unref(image);
	g_object_unref(mask);
	g_object_unref(signal_s);
}

static void
update_signal_bar(radio_gui* data, gint signal)
{
	GdkColor color;

	gtk_widget_show(data->signal_bar);
	gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR
		(data->signal_bar), signal / data->max_signal_strength);

	if (signal == 1)
		gdk_color_parse(COLOR_SIGNAL_LOW, &color);
	else if (signal == data->max_signal_strength)
		gdk_color_parse(COLOR_SIGNAL_HIGH, &color);
	else
		gdk_color_parse(COLOR_SIGNAL_MED, &color);

	gtk_widget_modify_bg(data->signal_bar, GTK_STATE_PRELIGHT, &color);
}

static gboolean
update_signal(radio_gui* data)
{
	if (!data->on || !data->show_signal)
	{
		data->signal_timeout_id = 0;
		gtk_widget_hide(data->signal_bar);

		if (data->on && !data->show_signal)
		{
			gtk_widget_hide(data->signal_image);
		}

		if (!data->on && !data->use_graphics)
		{
			gtk_widget_hide(data->signal_image);
		}

		return FALSE; // abort timer
	}
	else
	{
		double signal = radio_get_signal();

		// schedule the next call to update_signal
		if (data->signal_timeout_id == 0)
		{
			data->signal_timeout_id = g_timeout_add(500,
				(GtkFunction) update_signal, (gpointer) data);
		}

		if (data->use_graphics)
		{
			gtk_widget_hide(data->signal_bar);
			gtk_widget_show(data->signal_image);
			update_signal_image(data, signal);
		}
		else
		{
			gtk_widget_hide(data->signal_image);
			gtk_widget_show(data->signal_bar);
			update_signal_bar(data, signal);
		}
		return TRUE;
	}
}

#define make_preset_finder(name, arg, search_arg, get_col, get_var,	\
						free, comparison)	\
	static gboolean							\
	find_preset_by_##name(GtkTreeModel *model,			\
			search_arg,					\
			GtkTreeIter *search_iter)			\
	{								\
		arg;							\
		GtkTreeIter iter;					\
		gboolean valid;						\
		valid = gtk_tree_model_get_iter_first(model, &iter);	\
		while (valid)						\
		{							\
			gtk_tree_model_get(model, &iter,		\
					get_col, get_var, -1);		\
			if (comparison) {				\
				*search_iter = iter;			\
				return TRUE;				\
			}						\
			free;						\
			valid = gtk_tree_model_iter_next(model, &iter);	\
		}							\
		return FALSE;						\
	}								\

make_preset_finder(freq, gint freq, gint search_freq, 1, &freq, freq = 0,
							freq == search_freq);
make_preset_finder(name, gchar *name, const gchar *search_name, 0, &name,
				g_free(name), strcmp(name, search_name) == 0);

static void
update_tooltip(radio_gui* data)
{
	GtkWidget* ebox = data->ebox;
	GtkTreeIter iter;
	gchar *name;
	gchar *tooltip;

	DBG("Updating tooltip");

	GtkTreeModel *presets = GTK_TREE_MODEL(data->presets);
	if (find_preset_by_freq(presets, data->freq, &iter))
	{
		gtk_tree_model_get(presets, &iter, 0, &name, -1);
		tooltip = g_strdup_printf(_("Tuned to station %s"), name);
	}
	else
	{
		name = g_strdup_printf("%.1f", (float) data->freq / 100);
		tooltip = g_strdup_printf(_("Tuned to %s MHz"), name);
	}

	gtk_tooltips_set_tip(data->tooltips, ebox, tooltip, NULL);

	g_free(name);
	g_free(tooltip);
}

static void
update_label_when_radio_on(radio_gui *data)
{
	gchar *label;

	if (data->show_label)
		gtk_widget_show(data->label);
	else
		gtk_widget_hide(data->label);

	if (data->resolve_presets_in_label)
	{
		GtkTreeIter iter;
		GtkTreeModel *presets = GTK_TREE_MODEL(data->presets);
		if (find_preset_by_freq(presets, data->freq, &iter))
			gtk_tree_model_get(presets, &iter, 0, &label, -1);
		else
			label = g_strdup_printf
				("%5.1f", ((float) data->freq) / 100);
	}
	else
		label = g_strdup_printf("%5.1f", ((float) data->freq) / 100);

	gtk_label_set_label(GTK_LABEL(data->label), label);
	g_free(label);

	update_tooltip(data);
}

static void
update_label_when_radio_off(radio_gui *data)
{
	if (!data->use_graphics)
	{
		gtk_widget_hide(data->signal_image);
		gtk_widget_show(data->label);
		gtk_label_set_label(GTK_LABEL(data->label), _("- off -"));
	}
	else
	{
		gtk_widget_hide(data->label);
		update_signal_image(data, 0);
	}
	gtk_tooltips_set_tip(data->tooltips, data->ebox, _("Radio is off"),
									NULL);
}

static void
update_label(radio_gui *data)
{
	if (data->on)
		update_label_when_radio_on(data);
	else
		update_label_when_radio_off(data);
}

static gboolean
update_radio(radio_gui *data)
{
	// We could just remove the timer, but then we would have to make sure
	// to re-add it at all the appropriate places, thus we don't.
	if (!data->auto_update_display)
		return TRUE;

	if (radio_is_muted())
	{
		if (data->on)
		{
			// Radio is off, we thought it was on
			if (data->show_signal)
				gtk_widget_hide(data->signal_bar);
			data->on = FALSE;
		}
	}
	else
	{
		if (!data->on)
		{
			// Radio is on, we thought it was off
			gtk_tooltips_enable(data->tooltips);
			data->on = TRUE;
		}
		data->freq = radio_get_freq() * 100;
		DBG("Updating frequency to %d", data->freq);
	}
	update_label(data);
	update_signal(data);

	return TRUE;
}

static void
xfce4_radio_tune(radio_gui* data)
{
	double freq = data->freq / 100.0;
	radio_set_freq(freq);
	DBG("Tuning to %f", freq);

	update_label(data);
	update_signal(data);

	write_config(data, FALSE);
}

static gboolean
xfce4_radio_start(radio_gui* data)
{
	DBG("Starting %s", data->device);

	if (!radio_init(data->device, DRIVER_V4L2))
	{
		DBG("Failed!");
		GtkWindow* win = GTK_WINDOW(gtk_widget_get_toplevel(
							data->outer_box));
		GtkWidget* warn = gtk_message_dialog_new(win, 0,
			GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
			_("Error opening radio device"));
		gtk_dialog_run(GTK_DIALOG(warn));
		gtk_widget_destroy(warn);
		return FALSE;
	}

	if (strcmp(data->startup_command, "") != 0)
	{
		xfce_exec(data->startup_command, FALSE, FALSE, NULL);
	}
	xfce4_radio_tune(data);
	gtk_tooltips_enable(data->tooltips);
	return TRUE;
}

static void
xfce4_radio_stop(radio_gui* data)
{
	DBG("Stopping, auto_update_display=%d", data->auto_update_display);

	if (data->auto_update_display)
	{
		radio_mute();
	}
	else
	{
		radio_stop();
	}

	data->on = FALSE;
	gtk_tooltips_disable(data->tooltips);

	gtk_widget_hide(data->signal_bar);
	update_label(data);

	if (strcmp(data->shutdown_command, "") != 0)
	{
		xfce_exec(data->shutdown_command, FALSE, FALSE, NULL);
	}
}

static gint
parse_freq(const gchar *freq_char)
{
	gint mult, decimal_int;

	gint freq_int = 100 * atoi(freq_char);

	gchar *decimals = strstr(freq_char, ".");
	if (decimals)
		decimals++;
	else
	{
		// In some languages, a comma can be used instead of a point
		decimals = strstr(freq_char, ",");
		if (decimals)
			decimals++;
		else
			decimals = "0";
	}

	decimal_int = atoi(decimals);

	if (decimal_int < 10)
		mult = 10;
	else if (decimal_int < 100)
		mult = 1;
	else
		mult = 0;

	freq_int += mult * decimal_int;

	return freq_int;
}

static gboolean
parse_freq_and_tune(const char* freq_char, radio_gui* data)
{
	int freq_int = parse_freq(freq_char);
	DBG("Parsed frequency %d", freq_int);

	if (freq_int >= FREQ_MIN && freq_int <= FREQ_MAX)
	{
		data->freq = freq_int;
		xfce4_radio_tune(data);
		return TRUE;
	}
	return FALSE;
}

static void
radio_tune_gui(GtkEditable *menu_item, void *pointer)
{
	radio_gui* data = (radio_gui*) pointer;
	GtkWindow* win = GTK_WINDOW(gtk_widget_get_toplevel(data->outer_box));
	GtkWidget* dialog = gtk_dialog_new_with_buttons(_("Tune radio"),
				NULL, GTK_DIALOG_DESTROY_WITH_PARENT,
				GTK_STOCK_OK, GTK_RESPONSE_OK,
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, NULL);
	GtkWidget* box = GTK_DIALOG(dialog)->vbox;

	GtkWidget* label = gtk_label_new(_("Frequency [MHz]:"));
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);

	GtkWidget* freq = gtk_entry_new_with_max_length(5);
	gtk_widget_show(freq);
	gtk_box_pack_start(GTK_BOX(box), freq, FALSE, FALSE, 0);

	int retval;
	for (;;)
	{
		retval = gtk_dialog_run(GTK_DIALOG(dialog));

		if (	retval == GTK_RESPONSE_CANCEL ||
			retval == GTK_RESPONSE_DELETE_EVENT ||
			retval == GTK_RESPONSE_NONE) {
				break;
		}

		const char* freq_char = gtk_entry_get_text(GTK_ENTRY(freq));
		if (parse_freq_and_tune(freq_char, data)) break;

		GtkWidget* warn = gtk_message_dialog_new(win, 0,
					GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
						_("Illegal frequency."));
		gtk_dialog_run(GTK_DIALOG(warn));
		gtk_widget_destroy(warn);
	}
	gtk_widget_destroy(dialog);
}

static XfceRc *
get_config_rc()
{
	char *file;
	XfceRc *rc;

	// Take data from a clearly defined place RADIO_CONFIG
	file = xfce_resource_save_location(XFCE_RESOURCE_CONFIG, RADIO_CONFIG,
									TRUE);

	if (G_UNLIKELY(!file))
		return NULL;

	rc = xfce_rc_simple_open(file, FALSE);
	g_free(file);

	return rc;
}

static void
select_preset(GtkEditable* menu_item, radio_gui *data)
{
	GtkWidget* label = gtk_bin_get_child(GTK_BIN(menu_item));
	const gchar* text = gtk_label_get_text(GTK_LABEL(label));
	GtkTreeModel *presets = GTK_TREE_MODEL(data->presets);
	GtkTreeIter iter;

	if (find_preset_by_name(presets, text, &iter))
	{
		gtk_tree_model_get(presets, &iter, 1, &data->freq, -1);
		xfce4_radio_tune(data);
	}
}

static gboolean
mouse_click(GtkWidget* src, GdkEventButton *event, radio_gui* data)
{
	DBG("Mouse button %d clicked", event->button);
	if (event->button == 1)
	{
		if (data->on)
			xfce4_radio_stop(data);
		else
			data->on = xfce4_radio_start(data);
	}
	else if (event->button == 2 && data->on)
	{
		GtkWidget* menu = gtk_menu_new();
		GtkWidget* item = gtk_menu_item_new_with_label(_("Presets"));
		gtk_widget_show(item);
		gtk_menu_append(menu, item);
		gtk_widget_set_sensitive(item, FALSE);

		GtkWidget* separator = gtk_separator_menu_item_new();
		gtk_widget_show(separator);
		gtk_container_add(GTK_CONTAINER(menu), separator);
		gtk_widget_set_sensitive(separator, FALSE);

		gchar *name;
		GtkTreeModel *presets = GTK_TREE_MODEL(data->presets);
		GtkTreeIter iter;

		gboolean valid = gtk_tree_model_get_iter_first(presets, &iter);
		while (valid)
		{
			gtk_tree_model_get(presets, &iter, 0, &name, -1);
			item = gtk_menu_item_new_with_label(name);
			gtk_widget_show(item);
			g_free(name);
			gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
			g_signal_connect(GTK_WIDGET(item), "activate",
					G_CALLBACK(select_preset), data);
			valid = gtk_tree_model_iter_next(presets, &iter);
		}

		separator = gtk_separator_menu_item_new();
		gtk_widget_show(separator);
		gtk_container_add(GTK_CONTAINER(menu), separator);
		gtk_widget_set_sensitive(separator, FALSE);

		item = gtk_menu_item_new_with_label(_("Tune to frequency"));
		gtk_widget_show(item);
		gtk_menu_append(menu, item);
		g_signal_connect(GTK_WIDGET(item), "activate",
					G_CALLBACK(radio_tune_gui), data);

		gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
				event->button, event->time);
	}
	update_label(data);
	update_signal(data);
	return event->button != 3;
}

static void
mouse_scroll(GtkWidget* src, GdkEventScroll *event, radio_gui* data)
{
	GtkTreePath *path;
	GtkTreeIter iter, iter1;
	GtkTreeModel *presets;

	DBG("Mouse scrolled");
	if (!data->on)
		return;

	int direction = event->direction == GDK_SCROLL_UP ? -1 : 1;
	if (data->scroll == CHANGE_FREQ) {
		data->freq += direction * FREQ_STEP;
		if (data->freq > FREQ_MAX) data->freq = FREQ_MIN;
		if (data->freq < FREQ_MIN) data->freq = FREQ_MAX;
		xfce4_radio_tune(data);
	}
	else if (data->scroll == CHANGE_PRESET)
	{
		presets = GTK_TREE_MODEL(data->presets);
		if (!find_preset_by_freq(presets, data->freq, &iter))
		{
			// tune to first preset, if it exists
			if (!gtk_tree_model_get_iter_first(presets, &iter))
				return;

			gtk_tree_model_get(presets, &iter,
					1, &data->freq, -1);
			xfce4_radio_tune(data);
			return;
		}
		// preset found
		if (direction == 1)
		{
			if (!gtk_tree_model_iter_next(presets, &iter))
				return;
			gtk_tree_model_get(presets, &iter, 1, &data->freq, -1);
			xfce4_radio_tune(data);
		}
		else
		{
			path = gtk_tree_model_get_path(presets, &iter);

			if (!gtk_tree_path_prev(path))
				return;

			gtk_tree_model_get_iter(presets, &iter1, path);
			gtk_tree_path_free(path);
			gtk_tree_model_get(presets, &iter1, 1, &data->freq, -1);
			xfce4_radio_tune(data);
		}
	}
}

static void
add_boxes(radio_gui *gui)
{
	GtkWidget *align = gtk_alignment_new(0.5, 0.5, 1, 1);
	gtk_container_add(GTK_CONTAINER(gui->ebox), align);
	gtk_widget_show(align);

	if (gui->orientation == GTK_ORIENTATION_HORIZONTAL)
	{
		gui->outer_box = gtk_hbox_new(FALSE, 4);
		gtk_alignment_set_padding(GTK_ALIGNMENT(align), 0, 0, 4, 4);
	}
	else
	{
		gui->outer_box = gtk_vbox_new(FALSE, 0);
		gtk_alignment_set_padding(GTK_ALIGNMENT(align), 4, 4, 0, 0);
	}

	gtk_widget_show(gui->outer_box);
	gtk_container_add(GTK_CONTAINER(align), gui->outer_box);
}

static void
add_signal_bar(radio_gui *gui)
{
	GtkWidget *align = gtk_alignment_new(0, 0, 0, 0);
	gtk_container_add(GTK_CONTAINER(gui->outer_box), align);
	gtk_widget_show(align);

	if (gui->orientation == GTK_ORIENTATION_HORIZONTAL)
	{
		gtk_alignment_set_padding(GTK_ALIGNMENT(align),
			BORDER / 2, BORDER / 2, 0, 0);
	}
	else
	{
		gtk_alignment_set_padding(GTK_ALIGNMENT(align),
			0, 0, BORDER / 2, BORDER / 2);
	}

	GtkWidget *signal_bar = gtk_progress_bar_new();

	if (gui->orientation == GTK_ORIENTATION_HORIZONTAL)
		gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR
			(signal_bar), GTK_PROGRESS_BOTTOM_TO_TOP);
	else
		gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR
			(signal_bar), GTK_PROGRESS_LEFT_TO_RIGHT);

	gtk_container_add(GTK_CONTAINER(align), signal_bar);
	gui->signal_bar = signal_bar;
}

static void
add_label(radio_gui *gui)
{
	gui->label = gtk_label_new(NULL);
	gtk_container_add(GTK_CONTAINER(gui->outer_box), gui->label);
}

static void
add_image(radio_gui *gui)
{
	gui->signal_image = gtk_image_new();
	gtk_container_add(GTK_CONTAINER(gui->outer_box), gui->signal_image);
}

static void
add_gui_elements(radio_gui *gui)
{
	add_boxes(gui);
	add_label(gui);
	add_signal_bar(gui);
	add_image(gui);
}

static radio_gui *
create_gui(gint orientation)
{
	radio_gui* gui;
	gui = g_new(radio_gui, 1);
	gui->max_signal_strength = 3;
	gui->orientation = orientation;

	gui->ebox = gtk_event_box_new();
	gtk_widget_show(gui->ebox);

	g_signal_connect(GTK_WIDGET(gui->ebox), "button_press_event",
						G_CALLBACK(mouse_click), gui);
	g_signal_connect(GTK_WIDGET(gui->ebox), "scroll_event",
						G_CALLBACK(mouse_scroll), gui);
	add_gui_elements(gui);

	return gui;
}

static void
free_presets(radio_gui *data)
{
	g_object_unref(data->presets);
}

static void
radio_free(XfcePanelPlugin *plugin, radio_gui *data)
{
	if (data->on) xfce4_radio_stop(data);
	free_presets(data);
	g_free(data);

	if (data->signal_timeout_id)
		g_source_remove(data->signal_timeout_id);
	if (data->radio_timeout_id)
		g_source_remove(data->radio_timeout_id);
}

static GList *
radio_find_tuners(GList *tuners)
{
	DIR *dir;
	struct dirent *item;

	dir = opendir("/dev");
	if (dir == NULL)
	{
		perror("Could not open the directory");
		return tuners;
	}
	while ((item = readdir(dir)))
	{
		if (!strncmp(item->d_name, "radio", 5))
		{
			gchar *dev = g_strdup_printf("/dev/%s", item->d_name);
			tuners = g_list_append(tuners, dev);
		}
	}
	closedir(dir);
	return tuners;
}

static gchar *
radio_tuner_name(const gchar *devname)
{
	int fd = open(devname, O_RDONLY);

	struct v4l2_capability vcap;

	if (ioctl(fd, VIDIOC_QUERYCAP, &vcap) < 0)
	{
		perror("VIDIOC_QUERYCAP");
		return g_strdup_printf("Invalid tuner (%s)", devname);
	}
	else
	{
		return g_strdup_printf("%s (%s)", vcap.card, devname);
	}
}

static radio_gui *
plugin_control_new(XfcePanelPlugin *plugin)
{
	gint orientation = xfce_panel_plugin_get_orientation(plugin);
	radio_gui* plugin_data = create_gui(orientation);

	plugin_data->plugin = plugin;

	plugin_data->on = FALSE;
	plugin_data->freq = FREQ_INIT;

	// try to auto-detect the radio device
	GList *tuners = NULL;
	tuners = radio_find_tuners(tuners);

	if (g_list_length(tuners) > 0)
	{
		GList *tuner = g_list_first(tuners);
		strcpy(plugin_data->device, tuner->data);

		// free the list elements
		for (tuner = tuners; tuner; tuner = tuner->next)
			g_free(tuner->data);
	}
	else
	{
		// some default value
		strcpy(plugin_data->device, "/dev/radio");
	}
	g_list_free(tuners);

	plugin_data->show_signal = TRUE;
	plugin_data->use_graphics = TRUE;
	plugin_data->show_label = TRUE;
	plugin_data->auto_update_display = FALSE;
	plugin_data->presets = NULL;
	plugin_data->presets = NULL;
	plugin_data->scroll = CHANGE_FREQ;
	plugin_data->signal_timeout_id = 0;
	plugin_data->radio_timeout_id = g_timeout_add(2000,
			(GtkFunction)update_radio, (gpointer)plugin_data);
	plugin_data->tooltips = gtk_tooltips_new();
	plugin_data->startup_command[0] = '\0';
	plugin_data->shutdown_command[0] = '\0';

	gtk_container_add(GTK_CONTAINER(plugin), plugin_data->ebox);

	return plugin_data;
}

static void
radio_startup_command_changed(GtkEditable* editable, void *pointer)
{
	radio_gui* data = (radio_gui*) pointer;
	const char* command = gtk_entry_get_text(GTK_ENTRY(editable));
	strncpy(data->startup_command, command, MAX_COMMAND_LENGTH);
}

static void
radio_shutdown_command_changed(GtkEditable* editable, void *pointer)
{
	radio_gui* data = (radio_gui*) pointer;
	const char* command = gtk_entry_get_text(GTK_ENTRY(editable));
	strncpy(data->shutdown_command, command, MAX_COMMAND_LENGTH);
}

static void
radio_device_changed(GtkEditable* box, void *pointer)
{
	radio_gui* data = (radio_gui*) pointer;
	GtkComboBox *combo = GTK_COMBO_BOX(box);
	GtkTreeModel *model = gtk_combo_box_get_model(combo);

	GtkTreeIter iter;
	gtk_combo_box_get_active_iter(combo, &iter);

	GValue value={0,};

	gtk_tree_model_get_value(model, &iter, 0, &value);
	const gchar *device = g_value_get_string(&value);

	strncpy(data->device, device, MAX_DEVICE_NAME_LENGTH);
}

static void
radio_show_signal_changed(GtkEditable* editable, void *pointer)
{
	radio_gui* data = (radio_gui*) pointer;
	if (!data->show_label)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(editable),
			TRUE);
	data->show_signal = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON
								(editable));
	update_signal(data);
}

static void
radio_show_label_changed(GtkEditable* editable, void *pointer)
{
	radio_gui* data = (radio_gui*) pointer;
	if (!data->show_signal)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(editable),
			TRUE);
	data->show_label = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON
								(editable));
	update_label(data);
}

static void
radio_resolve_presets_changed(GtkEditable* editable, void *pointer)
{
	radio_gui* data = (radio_gui*) pointer;
	data->resolve_presets_in_label = gtk_toggle_button_get_active
		(GTK_TOGGLE_BUTTON(editable));
	update_label(data);
}

static void
radio_signal_type_changed(GtkEditable* editable, void *pointer)
{
	radio_gui* data = (radio_gui*) pointer;
	data->use_graphics = gtk_toggle_button_get_active
		(GTK_TOGGLE_BUTTON(editable));
	update_label(data);
	update_signal(data);
}

static void
radio_auto_update_display_changed(GtkEditable* editable, void *pointer)
{
	radio_gui* data = (radio_gui*) pointer;
	data->auto_update_display = gtk_toggle_button_get_active
						(GTK_TOGGLE_BUTTON(editable));
}

static void
radio_scroll_type_changed(GtkEditable* box, void *pointer)
{
	radio_gui* data = (radio_gui*) pointer;
	gint selection = gtk_combo_box_get_active(GTK_COMBO_BOX(box));
	data->scroll = selection == 0 ? CHANGE_FREQ : CHANGE_PRESET;
}

static void
radio_plugin_dialog_response(GtkWidget *dialog, int response, radio_gui *data)
{
	gtk_widget_destroy(dialog);
	xfce_panel_plugin_unblock_menu(data->plugin);
	write_config(data, TRUE);
}

static void
cell_float_to_text_cb(GtkTreeViewColumn *tree_column,
			GtkCellRenderer *cell,
			GtkTreeModel *tree_model,
			GtkTreeIter *iter,
			gpointer data)
{
	gint f;
	gchar *text;

	// Get the double value from the model
	gtk_tree_model_get(tree_model, iter, 1, &f, -1);
	// Now we can format the value ourselves
	text = g_strdup_printf("%.2f", f / 100.0);
	g_object_set(cell, "text", text, NULL);
	g_free(text);
}

static void
add_button_clicked_cb(GtkWidget *widget, GtkTreeSelection *selection)
{
	GtkTreeIter iter;
	GtkTreeModel *model;

	radio_gui *data = g_object_get_data(G_OBJECT(selection), "data");

	gtk_tree_selection_get_selected(selection, &model, &iter);
	gtk_list_store_append(GTK_LIST_STORE(model), &iter);
	gtk_list_store_set(GTK_LIST_STORE(model), &iter,
				0, _("unnamed"),
				1, data->freq, -1);
	gtk_tree_selection_select_iter(selection, &iter);
	write_config(data, TRUE);
}

static void
del_button_clicked_cb(GtkWidget *widget, GtkTreeSelection *selection)
{
	GtkTreeIter iter;
	GtkTreeModel *model;

	radio_gui *data = g_object_get_data(G_OBJECT(selection), "data");

	if (gtk_tree_selection_get_selected(selection, &model, &iter))
	{
		if (gtk_list_store_remove(GTK_LIST_STORE(model), &iter))
		{
			gtk_tree_selection_select_iter(selection,&iter);
		}
	}
	write_config(data, TRUE);
	update_tooltip(data);
}

static void
up_button_clicked_cb(GtkWidget *widget, GtkTreeSelection *selection)
{
	GtkTreeIter iter;
	GtkTreeModel *model;

	radio_gui *data = g_object_get_data(G_OBJECT(selection), "data");

	if (gtk_tree_selection_get_selected(selection, &model, &iter))
	{
		GtkTreePath *path = gtk_tree_model_get_path(model, &iter);
		if (gtk_tree_path_prev(path))
		{
			GtkTreeIter iter1 = iter;
			gtk_tree_model_get_iter(model, &iter1, path);
			gtk_list_store_swap(GTK_LIST_STORE(model),
					&iter, &iter1);
		}
		gtk_tree_path_free(path);
	}
	write_config(data, TRUE);
	// update_tooltip shows the first preset with the currently tuned
	// frequency. If you have multiple entries with the same frequency,
	// moving them up or down changes which one is shown in the tooltip
	update_tooltip(data);
}

static void
down_button_clicked_cb(GtkWidget *widget, GtkTreeSelection *selection)
{
	GtkTreeIter iter;
	GtkTreeModel *model;

	radio_gui *data = g_object_get_data(G_OBJECT(selection), "data");

	if (gtk_tree_selection_get_selected(selection, &model, &iter))
	{
		GtkTreeIter iter1 = iter;
		if (gtk_tree_model_iter_next(model, &iter1))
		{
			gtk_list_store_swap(GTK_LIST_STORE(model),
					&iter, &iter1);
		}
	}
	write_config(data, TRUE);
	// See up_button_clicked_cb
	update_tooltip(data);
}

static gboolean
list_view_key_press_event_cb(GtkWidget *widget, GdkEventKey *event,
							GtkTreeSelection *data)
{
	if (event->keyval == GDK_Delete)
		del_button_clicked_cb(widget, data);
	if (event->keyval == GDK_Insert)
		add_button_clicked_cb(widget, data);

	return FALSE;
}

static void
name_cell_edited_cb(GtkCellRendererText *cellrenderertext, gchar *path_str,
						gchar *new_val, radio_gui *data)
{
	GtkTreeIter iter;
	gboolean valid;

	GtkListStore *presets = data->presets;
	valid = gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(presets),
			&iter, path_str);
	g_return_if_fail(valid == TRUE);

	gtk_list_store_set(presets, &iter, 0, new_val, -1);
	write_config(data, TRUE);
	update_tooltip(data);
}

static void
freq_cell_edited_cb(GtkCellRendererText *cellrenderertext, gchar *path_str,
						gchar *new_val, radio_gui *data)
{
	GtkTreeIter iter;
	gboolean valid;
	gint value;

	value = parse_freq(new_val);

	if (value < FREQ_MIN) value = FREQ_MIN;
	if (value > FREQ_MAX) value = FREQ_MAX;

	GtkListStore *presets = data->presets;
	valid = gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(presets),
			&iter, path_str);
	g_return_if_fail(valid == TRUE);

	gtk_list_store_set(presets, &iter, 1, value, -1);
	write_config(data, TRUE);
	update_tooltip(data);
}

static void
choose_command(GtkWidget *dialog)
{
	GtkWidget *dlg;
	gchar *command;
	const gchar *prev;

	dlg = gtk_file_chooser_dialog_new(_("Select command"),
				GTK_WINDOW(gtk_widget_get_toplevel(dialog)),
				GTK_FILE_CHOOSER_ACTION_OPEN,
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				GTK_STOCK_OPEN, GTK_RESPONSE_OK, NULL);

	prev = gtk_entry_get_text(GTK_ENTRY(dialog));
	if (!strcmp(prev, "")) prev = "~"; // open home directory if not set
	gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(dlg), prev);

	if (G_LIKELY(gtk_dialog_run(GTK_DIALOG(dlg)) == GTK_RESPONSE_OK))
	{
		command = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dlg));
		gtk_entry_set_text(GTK_ENTRY(dialog), command);
		g_free(command);
	}
	gtk_widget_destroy(dlg);
}

static void
radio_plugin_create_options(XfcePanelPlugin *plugin, radio_gui *data)
{
	GtkWidget *label;
	GtkWidget *hbox;
	GtkWidget *dialog;
	GtkWidget *frame;
	GtkWidget *real_frame;

	GtkWidget *show_signal;			// show the signal when on
	GtkWidget *show_label;			// show the label when on
	GtkWidget *graphical_signal_strength;	// show graphical signal level
	GtkWidget *sync_state;			// sync state with the card
	GtkWidget *startup_command;		// post-down command
	GtkWidget *shutdown_command;		// post-down command
	GtkWidget *radio_dev;			// v4l device
	GtkWidget *scrolling;			// mouse-scrolling action
	GtkWidget *preset_box;
	GtkWidget *button_box;
	GtkWidget *notebook;
	GtkWidget *scrolled_window;
	GtkWidget *list_view;
	GtkWidget *current_page;
	GtkWidget *align;

	GtkTreeSelection *selection;
	GtkCellRenderer *cellrenderer;
	GtkTreeViewColumn *list_column;

	gchar *title;

	xfce_panel_plugin_block_menu(plugin);

	dialog = xfce_titled_dialog_new_with_buttons(_("Radio"),
		GTK_WINDOW(gtk_widget_get_toplevel(GTK_WIDGET(plugin))),
		GTK_DIALOG_DESTROY_WITH_PARENT |
		GTK_DIALOG_NO_SEPARATOR,
		GTK_STOCK_CLOSE, GTK_RESPONSE_OK,
		NULL
	);

	xfce_titled_dialog_set_subtitle(XFCE_TITLED_DIALOG(dialog),
					_("Configure the radio plugin"));
	gtk_window_set_icon_name(GTK_WINDOW(dialog), "xfce-radio");
	gtk_container_set_border_width(GTK_CONTAINER(dialog), 2);

	g_signal_connect(dialog, "response",
			G_CALLBACK(radio_plugin_dialog_response), data);

	list_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(data->presets));
	gtk_widget_set_size_request(list_view, 200, 150);
	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(list_view), FALSE);
	gtk_widget_show(list_view);

	cellrenderer = gtk_cell_renderer_text_new();
	g_object_set(G_OBJECT(cellrenderer),
		"mode", GTK_CELL_RENDERER_MODE_EDITABLE,
		"editable", TRUE, NULL);
	list_column = gtk_tree_view_column_new_with_attributes
					(NULL, cellrenderer, "text", 0, NULL);
	gtk_tree_view_column_set_min_width(list_column, 300);
	gtk_tree_view_append_column(GTK_TREE_VIEW(list_view), list_column);
	g_signal_connect(GTK_OBJECT(cellrenderer), "edited",
			GTK_SIGNAL_FUNC(name_cell_edited_cb), (gpointer)data);

	cellrenderer = gtk_cell_renderer_text_new();
	g_object_set(G_OBJECT(cellrenderer),
		"mode", GTK_CELL_RENDERER_MODE_EDITABLE,
		"editable", TRUE, \
		"xalign", 1.0f, NULL);
	list_column = gtk_tree_view_column_new_with_attributes
			(NULL, cellrenderer, "text", 1, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(list_view), list_column);
	gtk_tree_view_column_set_cell_data_func(list_column, cellrenderer,
						cell_float_to_text_cb,
						NULL, NULL);
	g_signal_connect(GTK_OBJECT(cellrenderer), "edited",
			GTK_SIGNAL_FUNC(freq_cell_edited_cb), (gpointer)data);

	button_box = gtk_hbox_new(FALSE, 0);

	GtkWidget *add_button  = gtk_button_new_from_stock(GTK_STOCK_ADD);
	GtkWidget *del_button  = gtk_button_new_from_stock(GTK_STOCK_DELETE);
	GtkWidget *up_button   = gtk_button_new_from_stock(GTK_STOCK_GO_UP);
	GtkWidget *down_button = gtk_button_new_from_stock(GTK_STOCK_GO_DOWN);

	gtk_box_pack_end(GTK_BOX(button_box), del_button, FALSE, FALSE, 0);
	gtk_box_pack_end(GTK_BOX(button_box), add_button, FALSE, FALSE, 0);
	gtk_box_pack_end(GTK_BOX(button_box), up_button,  FALSE, FALSE, 0);
	gtk_box_pack_end(GTK_BOX(button_box), down_button,FALSE, FALSE, 0);

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(list_view));
	gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);

	g_object_set_data(G_OBJECT(selection), "data", data);

	g_signal_connect(GTK_OBJECT(add_button), "clicked",
		GTK_SIGNAL_FUNC( add_button_clicked_cb), selection);
	g_signal_connect(GTK_OBJECT(del_button), "clicked",
		GTK_SIGNAL_FUNC( del_button_clicked_cb), selection);
	g_signal_connect(GTK_OBJECT(up_button), "clicked",
		GTK_SIGNAL_FUNC(up_button_clicked_cb), selection);
	g_signal_connect(GTK_OBJECT(down_button), "clicked",
		GTK_SIGNAL_FUNC(down_button_clicked_cb), selection);
	g_signal_connect(GTK_OBJECT(list_view), "key-press-event",
		GTK_SIGNAL_FUNC(list_view_key_press_event_cb), selection);

	scrolled_window = gtk_scrolled_window_new(NULL, NULL);
	gtk_container_add(GTK_CONTAINER(scrolled_window), list_view);
	gtk_scrolled_window_set_shadow_type(
					GTK_SCROLLED_WINDOW(scrolled_window),
					GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_widget_show(scrolled_window);

	preset_box = gtk_vbox_new(FALSE, 4);
	gtk_box_pack_start(GTK_BOX(preset_box), scrolled_window,
			TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(preset_box), button_box,
			FALSE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(preset_box), 4);
	gtk_widget_show_all(preset_box);

	// The tabbed "notebook"
	notebook = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(notebook), TRUE);
	gtk_container_set_border_width (GTK_CONTAINER (notebook), BORDER-3);

	// Add the presets page
	label = gtk_label_new(_("Presets"));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), preset_box, label);
	gtk_box_pack_start
		(GTK_BOX(GTK_DIALOG(dialog)->vbox), notebook, TRUE, TRUE, 0);
	gtk_widget_show(notebook);

	// Add the UI options page
	align = gtk_alignment_new(0, 0, 0.5, 0.5);
	gtk_widget_show(align);
	gtk_alignment_set_padding(GTK_ALIGNMENT(align), 0, 5, 5, 5);

	label = gtk_label_new(_("User interface"));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), align, label);

	current_page = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(current_page);
	gtk_container_add(GTK_CONTAINER(align), current_page);

	// - Signal strength
	real_frame = gtk_frame_new(_("Signal strength"));
	gtk_box_pack_start(GTK_BOX(current_page), real_frame, FALSE, FALSE, 9);
	gtk_widget_show(real_frame);

	align = gtk_alignment_new(0, 0, 1, 1);
	gtk_alignment_set_padding(GTK_ALIGNMENT(align), 0, 2, 2, 2);
	gtk_container_add(GTK_CONTAINER(real_frame), align);
	gtk_widget_show(align);

	frame = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(align), frame);
	gtk_widget_show(frame);

	// -- Show the signal strength
	show_signal = gtk_check_button_new_with_label
		(_("Show the signal strength indicator"));
	gtk_widget_show(show_signal);
	gtk_container_add(GTK_CONTAINER(frame), show_signal);

	// -- Use graphics
	graphical_signal_strength = gtk_check_button_new_with_label
		(_("Use graphics instead of text labels and progress bars"));
	gtk_widget_show(graphical_signal_strength);
	gtk_container_add(GTK_CONTAINER(frame), graphical_signal_strength);

	// - Current station
	real_frame = gtk_frame_new(_("Current station"));
	gtk_box_pack_start(GTK_BOX(current_page), real_frame, FALSE, FALSE, 9);
	gtk_widget_show(real_frame);

	align = gtk_alignment_new(0, 0, 1, 1);
	gtk_alignment_set_padding(GTK_ALIGNMENT(align), 0, 2, 2, 2);
	gtk_container_add(GTK_CONTAINER(real_frame), align);
	gtk_widget_show(align);

	frame = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(align), frame);
	gtk_widget_show(frame);

	// -- Show the label
	show_label = gtk_check_button_new_with_label
		(_("Show the current station"));
	gtk_widget_show(show_label);
	gtk_container_add(GTK_CONTAINER(frame), show_label);

	// -- Resolve presets
	GtkWidget *resolve_presets = gtk_check_button_new_with_label
		(_("Show preset names instead of frequencies"));
	gtk_widget_show(resolve_presets);
	gtk_container_add(GTK_CONTAINER(frame), resolve_presets);

	// - Mouse
	real_frame = gtk_frame_new(_("Mouse interaction"));
	gtk_box_pack_start(GTK_BOX(current_page), real_frame, FALSE, FALSE, 9);
	gtk_widget_show(real_frame);

	align = gtk_alignment_new(0, 0, 1, 1);
	gtk_alignment_set_padding(GTK_ALIGNMENT(align), 0, 2, 2, 2);
	gtk_container_add(GTK_CONTAINER(real_frame), align);
	gtk_widget_show(align);

	frame = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(align), frame);
	gtk_widget_show(frame);

	// -- Mouse-scrolling
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), hbox);
	gtk_widget_show(hbox);

	label = gtk_label_new(_("Mouse scrolling changes"));
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	GtkTreeIter iter;
	GtkListStore *model = gtk_list_store_new(1, G_TYPE_STRING);
	GtkCellRenderer *renderer;

	scrolling = gtk_combo_box_new_with_model(GTK_TREE_MODEL(model));
	renderer = gtk_cell_renderer_text_new();
	gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(scrolling), renderer, TRUE);
	gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(scrolling), renderer,
			"text", 0, NULL);

	gtk_list_store_append(model, &iter);
	gtk_list_store_set(model, &iter, 0, _("frequency"), -1);
	gtk_list_store_append(model, &iter);
	gtk_list_store_set(model, &iter, 0, _("station preset"), -1);

	gtk_box_pack_start(GTK_BOX(hbox), scrolling, FALSE, FALSE, 5);
	gtk_widget_show(scrolling);

	// Add the device options page
	align = gtk_alignment_new(0, 0, 0.5, 0.5);
	gtk_widget_show(align);
	gtk_alignment_set_padding(GTK_ALIGNMENT(align), 0, 5, 5, 5);

	label = gtk_label_new(_("Radio device"));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), align, label);

	current_page = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(current_page);
	gtk_container_add(GTK_CONTAINER(align), current_page);

	// - Radio device
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(current_page), hbox, FALSE, FALSE, 9);
	gtk_widget_show(hbox);

	label = gtk_label_new(_("Radio device:"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	model = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);

	radio_dev = gtk_combo_box_new_with_model(GTK_TREE_MODEL(model));
	gtk_box_pack_start(GTK_BOX(hbox), radio_dev, FALSE, FALSE, 5);
	gtk_widget_show(radio_dev);

	renderer = gtk_cell_renderer_text_new();
	gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(radio_dev), renderer, TRUE);
	gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(radio_dev), renderer,
							"text", 1, NULL);

	GList *tuners = NULL, *tuner;
	tuners = radio_find_tuners(tuners);
	gchar *devname;

	for (tuner = tuners; tuner; tuner = tuner->next)
	{
		devname = tuner->data;
		gtk_list_store_append(model, &iter);
		gtk_list_store_set(model, &iter,
				0, devname,
				1, radio_tuner_name(devname),
		-1);

		if (!g_strcmp0(devname, data->device))
		{
			gtk_combo_box_set_active_iter(GTK_COMBO_BOX(radio_dev),
					&iter);
		}

		g_free(tuner->data);
	}
	g_list_free(tuners);

	// - Synchronize state
	sync_state = gtk_check_button_new_with_label
					(_("Synchronize state with the card"));

	gtk_widget_show(sync_state);

	gtk_box_pack_start(GTK_BOX(current_page), sync_state, FALSE, FALSE, 0);

	frame = gtk_alignment_new(0, 0, 0, 0);
	gtk_alignment_set_padding(GTK_ALIGNMENT(frame), 0, 0, 20, 0);
	gtk_widget_show(frame);

	title = g_strdup_printf("<i>%s</i>", _("This watches your radio card "
		"for changes done by other\napplications. Note that this may "
		"cause issues with some\ntuner cards."));
	label = gtk_label_new("");
	gtk_label_set_markup(GTK_LABEL(label), title);
	g_free(title);

	gtk_widget_show(label);
	gtk_container_add(GTK_CONTAINER(frame), label);
	gtk_box_pack_start(GTK_BOX(current_page), frame, FALSE, FALSE, 0);

	// Add the commands page
	align = gtk_alignment_new(0, 0, 0.5, 0.5);
	gtk_widget_show(align);
	gtk_alignment_set_padding(GTK_ALIGNMENT(align), 0, 5, 5, 5);

	label = gtk_label_new(_("Commands"));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), align, label);

	current_page = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(current_page);
	gtk_container_add(GTK_CONTAINER(align), current_page);

	// -- Post-startup command
	align = gtk_alignment_new(0, 0, 0, 0);
	gtk_box_pack_start(GTK_BOX(current_page), align, FALSE, FALSE, 9);
	gtk_widget_show(align);

	title = g_strdup_printf("<b>%s</b>", _("Command to run after startup"));
	label = gtk_label_new(NULL);
	gtk_label_set_markup(GTK_LABEL(label), title);
	gtk_container_add(GTK_CONTAINER(align), label);
	gtk_widget_show(label);
	g_free(title);

	align = gtk_alignment_new(0, 0, 1, 1);
	gtk_alignment_set_padding(GTK_ALIGNMENT(align), 0, 0, 20, 0);
	gtk_box_pack_start(GTK_BOX(current_page), align, FALSE, TRUE, 0);
	gtk_widget_show(align);

	hbox = gtk_hbox_new(FALSE, 12);
	gtk_container_add(GTK_CONTAINER(align), hbox);
	gtk_widget_show(hbox);

	startup_command = gtk_entry_new_with_max_length(MAX_COMMAND_LENGTH);
	gtk_entry_set_text(GTK_ENTRY(startup_command), data->startup_command);
	gtk_box_pack_start(GTK_BOX(hbox), startup_command, TRUE, TRUE, 0);
	gtk_widget_show(startup_command);

	GtkWidget *button = gtk_button_new_from_stock(GTK_STOCK_OPEN);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, TRUE, 0);
	gtk_widget_show(button);

	g_signal_connect_swapped(button, "clicked",
				G_CALLBACK(choose_command), startup_command);

	// -- Post-shutdown command
	align = gtk_alignment_new(0, 0, 0, 0);
	gtk_box_pack_start(GTK_BOX(current_page), align, FALSE, FALSE, 9);
	gtk_widget_show(align);

	title = g_strdup_printf("<b>%s</b>",_("Command to run after shutdown"));
	label = gtk_label_new(NULL);
	gtk_label_set_markup(GTK_LABEL(label), title);
	gtk_container_add(GTK_CONTAINER(align), label);
	gtk_widget_show(label);
	g_free(title);

	align = gtk_alignment_new(0, 0, 1, 1);
	gtk_alignment_set_padding(GTK_ALIGNMENT(align), 0, 0, 20, 0);
	gtk_box_pack_start(GTK_BOX(current_page), align, FALSE, TRUE, 0);
	gtk_widget_show(align);

	hbox = gtk_hbox_new(FALSE, 12);
	gtk_container_add(GTK_CONTAINER(align), hbox);
	gtk_widget_show(hbox);

	shutdown_command = gtk_entry_new_with_max_length(MAX_COMMAND_LENGTH);
	gtk_entry_set_text(GTK_ENTRY(shutdown_command), data->shutdown_command);
	gtk_box_pack_start(GTK_BOX(hbox), shutdown_command, TRUE, TRUE, 0);
	gtk_widget_show(shutdown_command);

	button = gtk_button_new_from_stock(GTK_STOCK_OPEN);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, TRUE, 0);
	gtk_widget_show(button);

	g_signal_connect_swapped(button, "clicked",
				G_CALLBACK(choose_command), shutdown_command);

	// set values
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
		(show_signal), data->show_signal);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
		(graphical_signal_strength), data->use_graphics);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
		(show_label), data->show_label);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
		(resolve_presets), data->resolve_presets_in_label);
	gtk_combo_box_set_active(GTK_COMBO_BOX
		(scrolling), data->scroll == CHANGE_FREQ ? 0 : 1);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
		(sync_state), data->auto_update_display);

	// Connect signals
	g_signal_connect((gpointer) startup_command, "changed",
			G_CALLBACK(radio_startup_command_changed), data);
	g_signal_connect((gpointer) shutdown_command, "changed",
			G_CALLBACK(radio_shutdown_command_changed), data);
	g_signal_connect((gpointer) radio_dev, "changed",
			G_CALLBACK(radio_device_changed), data);
	g_signal_connect(G_OBJECT(show_signal), "toggled",
			G_CALLBACK(radio_show_signal_changed), data);
	g_signal_connect(G_OBJECT(show_label), "toggled",
			G_CALLBACK(radio_show_label_changed), data);
	g_signal_connect(G_OBJECT(graphical_signal_strength), "toggled",
			G_CALLBACK(radio_signal_type_changed), data);
	g_signal_connect(G_OBJECT(resolve_presets), "toggled",
			G_CALLBACK(radio_resolve_presets_changed), data);
	g_signal_connect(G_OBJECT(sync_state), "toggled",
			G_CALLBACK(radio_auto_update_display_changed), data);
	g_signal_connect(G_OBJECT(scrolling), "changed",
			G_CALLBACK(radio_scroll_type_changed), data);

	gtk_widget_show(dialog);
}

static void
radio_plugin_set_size(XfcePanelPlugin *plugin, int size, radio_gui *data)
{
	GtkWidget* bar = data->signal_bar;

	if (data->orientation == GTK_ORIENTATION_HORIZONTAL)
		gtk_widget_set_size_request(bar, BORDER, size - BORDER);
	else
		gtk_widget_set_size_request(bar, size - BORDER, BORDER);
}

static void
radio_set_orientation(XfcePanelPlugin *plugin, GtkOrientation orientation,
								radio_gui *gui)
{
	gui->orientation = orientation;
	gtk_container_remove(GTK_CONTAINER(gui->ebox), gui->outer_box->parent);
	add_gui_elements(gui);

	update_label(gui);
}

static void
write_config_signal(XfcePanelPlugin *plugin, radio_gui *data)
{
	write_config(data, TRUE);
}

static void
write_config(radio_gui *data, gboolean save_presets)
{
	XfceRc *rc;
	int i;

	if (!(rc = get_config_rc()))
		return;

	xfce_rc_set_group	(rc, "radio plugin");

	xfce_rc_write_entry	(rc, "dev", data->device);
	xfce_rc_write_entry	(rc, "startup_cmd", data->startup_command);
	xfce_rc_write_entry	(rc, "shutdown_cmd", data->shutdown_command);

	xfce_rc_write_int_entry	(rc, "frq", data->freq);
	xfce_rc_write_int_entry	(rc, "scroll", data->scroll);
	xfce_rc_write_bool_entry(rc, "show_signal", data->show_signal);
	xfce_rc_write_bool_entry(rc, "update_display",
						data->auto_update_display);
	xfce_rc_write_bool_entry(rc, "show_label", data->show_label);
	xfce_rc_write_bool_entry(rc, "resolve_presets_in_label",
						data->resolve_presets_in_label);
	xfce_rc_write_bool_entry(rc, "graphical_signal",
						data->use_graphics);

	if (!save_presets)
	{
		xfce_rc_close(rc);
		return;
	}

	gchar *name;
	gint freq;
	GtkTreeIter iter;
	GtkTreeModel *presets = GTK_TREE_MODEL(data->presets);

	DBG("Writing configuration");
	gboolean valid = gtk_tree_model_get_iter_first(presets, &iter);

	// We start at 10 since there seems to be a problem loading keys which
	// are only 1 char long
	i = 10;
	while (valid)
	{
		if (i == 10)
		{
			xfce_rc_delete_group(rc, PRESET_NAMES, FALSE);
			xfce_rc_delete_group(rc, PRESET_FREQS, FALSE);
		}

		gtk_tree_model_get(presets, &iter, 0, &name, 1, &freq, -1);
		gchar *key = g_strdup_printf("%d", i);
		DBG("freq=%d, name=%s", freq, name);

		xfce_rc_set_group(rc, PRESET_NAMES);
		xfce_rc_write_entry(rc, key, name);
		xfce_rc_set_group(rc, PRESET_FREQS);
		xfce_rc_write_int_entry(rc, key, freq);

		g_free(name);
		g_free(key);
		i++;
		valid = gtk_tree_model_iter_next(presets, &iter);
	}
	xfce_rc_close(rc);
}

static void
read_config(XfcePanelPlugin *plugin, radio_gui *data)
{
	const char *value;
	XfceRc *rc;

	if (!(rc = get_config_rc()))
		return;

	xfce_rc_set_group(rc, "radio plugin");

	data->freq = xfce_rc_read_int_entry
					(rc, "frq", FREQ_INIT);
	data->scroll = xfce_rc_read_int_entry
					(rc, "scroll", CHANGE_FREQ);
	data->show_signal = xfce_rc_read_bool_entry
					(rc, "show_signal", TRUE);
	data->auto_update_display = xfce_rc_read_bool_entry
					(rc, "update_display", TRUE);
	data->show_label = xfce_rc_read_bool_entry
					(rc, "show_label", TRUE);
	data->resolve_presets_in_label = xfce_rc_read_bool_entry
					(rc, "resolve_presets_in_label", TRUE);
	data->use_graphics = xfce_rc_read_bool_entry
					(rc, "graphical_signal", TRUE);

	if ((value = xfce_rc_read_entry(rc, "dev", NULL)) && *value)
		strncpy(data->device, value, MAX_DEVICE_NAME_LENGTH);

	if ((value = xfce_rc_read_entry(rc, "startup_cmd", NULL)) && *value)
		strncpy(data->startup_command, value, MAX_COMMAND_LENGTH);

	if ((value = xfce_rc_read_entry(rc, "shutdown_cmd", NULL)) && *value)
		strncpy(data->shutdown_command, value, MAX_COMMAND_LENGTH);

	GtkTreeIter iter;
	g_return_if_fail(data->presets == NULL);
	data->presets = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_INT);
	GtkListStore *presets = data->presets;

	gchar** names = xfce_rc_get_entries(rc, PRESET_NAMES);
	gchar** freqs = xfce_rc_get_entries(rc, PRESET_FREQS);

	if (names == NULL || freqs == NULL) {
		xfce_rc_close(rc);
		update_label(data);
		return;
	}
	gchar **ename = names;
	gchar **efreq = freqs;

	while (*ename || *efreq)
	{
		xfce_rc_set_group(rc, PRESET_NAMES);
		const gchar *vname = xfce_rc_read_entry(rc, *ename,
							_(DEFAULT_NAME));
		xfce_rc_set_group(rc, PRESET_FREQS);
		gint vfreq = xfce_rc_read_int_entry(rc, *efreq, FREQ_MIN);
		DBG("freq=%d, name=%s", vfreq, vname);

		gtk_list_store_append(presets, &iter);
		gtk_list_store_set(presets, &iter, 0, vname, 1, vfreq, -1);
		ename++; efreq++;
	}
	g_strfreev(names);
	g_strfreev(freqs);
	xfce_rc_close(rc);

	update_label(data);
}

static void
radio_plugin_construct(XfcePanelPlugin *plugin)
{
	xfce_textdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR, "UTF-8");

	radio_gui *data = plugin_control_new(plugin);
	read_config(plugin, data);

	xfce_panel_plugin_menu_show_configure(plugin);

	g_signal_connect(plugin, "configure-plugin",
			G_CALLBACK(radio_plugin_create_options), data);
	g_signal_connect(plugin, "size-changed", G_CALLBACK
						(radio_plugin_set_size), data);
	g_signal_connect(plugin, "free-data", G_CALLBACK(radio_free), data);
	g_signal_connect(plugin, "save", G_CALLBACK(write_config_signal),
									data);
	g_signal_connect(plugin, "orientation-changed",
				G_CALLBACK(radio_set_orientation), data);
}

XFCE_PANEL_PLUGIN_REGISTER_EXTERNAL(radio_plugin_construct);
